### Version 1.4.3 ###
### Released August 2025 ###


import arcpy
import os
import datetime
import math
import time
arcpy.env.maintainCurveSegments = True
    
class Toolbox(object):
    def __init__(self):
        """Define the toolbox (the name of the toolbox is the name of the
        .pyt file)."""
        self.label = "Toolbox"
        self.alias = "QC_Tools"

        # List of tool classes associated with this toolbox
        self.tools = [QAQCcodeReset_poly,
                      sliverWetlands_poly,
                      overlappingWetlands_poly,
                      wetlandTypeCalc_poly,
                      lakeAndPondSize,
                      adjacentWetlands,
                      QAQCSummary_poly,
                      sliverUplands_poly,
                      incorrectWetlandsCodes_poly,
                      combinedPolyTools,
                      QAQCcodeReset_line,
                      wetlandTypeCalc_line,
                      sliverWetlands_line,
                      incorrectWetlandsCodes_line,
                      linearOverlaps,
                      linearPolyOverlaps_line,
                      linearGaps_line,
                      linearDataPrep,
                      combinedLineTools,
                      QAQCSummary_line,
                      combinedPolyandLineTools,
                      adjacentRiparian,
                      overlappingRiparian,
                      sliverRiparian,
                      sliverUplands_Riparian,
                      riparianTypeCalc,
                      incorrectRiparianCodes,
                      QAQCcodeReset_Riparian,
                      QAQCSummary_Riparian,
                      combinedRiparianTools,
                      combinedWetlandRiparianTools,
                      create_NWIID,
                      create_NWIID_rip,
                      create_NWIID_line
                      ]


class QAQCcodeReset_poly(object):
    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "QAQC Code Reset - Polygonal"
        self.description = "This tool calculates the QAQC_Code to be 'NNNNNN'. \n This erases all recorded errors in the dataset and properly attributes the field for use by all other models. "
        self.canRunInBackground = False
        self.category = "Individual Polygonal Tools"

    
    def getParameterInfo(self):
        """Define parameter definitions"""
        # params [0]
        input_gdb = arcpy.Parameter(name = "input_gdb",
                                   displayName = "Input Geodatabase",
                                   datatype = "DEWorkspace",
                                   parameterType = "Required", #Required|Optional|Derived
                                   direction = "Input", #Input|Output
                                   )

        work_area = arcpy.Parameter(name = "work_area",
                                     displayName = "Work Area",
                                     datatype = "String",
                                     parameterType = "Required", #Required|Optional|Derived
                                     direction = "Input", #Input|Output
                                     )
        work_area.value = "CONUS"
        work_area.filter.type = "ValueList"
        work_area.filter.list = ["CONUS","AK","HI","PRVI","PacTrust"]

        #params = None
        return [input_gdb,work_area]

    def isLicensed(self):
        """Set whether tool is licensed to execute."""
        return True

    def updateParameters(self, parameters):
        """Modify the values and properties of parameters before internal
        validation is performed.  This method is called whenever a parameter
        has been changed."""

        return

    def updateMessages(self, parameters):
        """Modify the messages created by internal validation for each tool
        parameter.  This method is called after internal validation."""
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        if not arcpy.Exists(wet_poly):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_poly.".format(work_area=work_area))
        

        return

    def execute(self, parameters, messages):
        """The source code of the tool."""

        arcpy.AddMessage("QAQC Code Reset Version 1.4.3")
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        
        arcpy.management.Compact(input_gdb)

        start_count = arcpy.management.GetCount(wet_poly)
        QAQCcodeReset_poly.QAQCcodeReset_tool(input_gdb,work_area)
        end_count = arcpy.management.GetCount(wet_poly)
        if start_count[0] != end_count[0]:
            arcpy.AddWarning("The number of features in {work_area}_wet_poly changed during the course of this tool.".format(work_area=work_area))
            
        return

    @staticmethod
    def QAQCcodeReset_tool(arg1,arg2):
        input_gdb = str(arg1)
        work_area = str(arg2)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))

        qaqc_code = "'NNNNNN'"

        edit = arcpy.da.Editor(input_gdb)
        edit.startEditing()
        edit.startOperation()
        arcpy.management.CalculateField(wet_poly,"QAQC_CODE",qaqc_code,"PYTHON_9.3")
        edit.stopOperation()
        edit.stopEditing(True)

        return



class sliverWetlands_poly(object):
    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "Sliver Wetlands - Polygonal"
        self.description = "This tool identifies wetland polygons less than 0.01 acres, and changes character 3 of QAQC_Code = 'S'. \n These wetland features fall below the minimum mapping standard for wetlands and should be reviewed. Actual wetland features flagged as sliver wetlands can be justified as correct in the comments field of the QAQC_Summary table. These comments will only be saved if the ‘Append to History Table’ box is checked prior to running the All_QAQC_Checks model."
        self.canRunInBackground = False
        self.category = "Individual Polygonal Tools"

    
    def getParameterInfo(self):
        """Define parameter definitions"""
        # params [0]
        input_gdb = arcpy.Parameter(name = "input_gdb",
                                   displayName = "Input Geodatabase",
                                   datatype = "DEWorkspace",
                                   parameterType = "Required", #Required|Optional|Derived
                                   direction = "Input", #Input|Output
                                   )

        work_area = arcpy.Parameter(name = "work_area",
                                     displayName = "Work Area",
                                     datatype = "String",
                                     parameterType = "Required", #Required|Optional|Derived
                                     direction = "Input", #Input|Output
                                     )
        work_area.value = "CONUS"
        work_area.filter.type = "ValueList"
        work_area.filter.list = ["CONUS","AK","HI","PRVI","PacTrust"]

        #params = None
        
        return [input_gdb, work_area]
    
    def isLicensed(self):
        """Set whether tool is licensed to execute."""
        return True

    def updateParameters(self, parameters):
        """Modify the values and properties of parameters before internal
        validation is performed.  This method is called whenever a parameter
        has been changed."""
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        if not arcpy.Exists(wet_poly):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_poly.".format(work_area=work_area))

        return

    def updateMessages(self, parameters):
        """Modify the messages created by internal validation for each tool
        parameter.  This method is called after internal validation."""
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        if not arcpy.Exists(wet_poly):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_poly.".format(work_area=work_area))

        return

    def execute(self, parameters, messages):
        """The source code of the tool."""

        arcpy.AddMessage("Sliver Wetlands Version 1.4.3")
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))

        arcpy.management.Compact(input_gdb)
        
        start_count = arcpy.management.GetCount(wet_poly)
        arcpy.AddMessage("Begin Sliver Wetlands tool")
        sliverWetlands_poly.sliverWetlands(input_gdb,work_area)
        end_count = arcpy.management.GetCount(wet_poly)
        if start_count[0] != end_count[0]:
            arcpy.AddWarning("The number of features in {work_area}_wet_poly changed during the course of this tool.".format(work_area=work_area))
        
        return

    @staticmethod
    def sliverWetlands(arg1,arg2):
        input_gdb = str(arg1)
        work_area = str(arg2)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))

        qaqc_code = "'NNNNNN'"
        
        arcpy.CalculateField_management(wet_poly,"ACRES","!shape.area@acres!","PYTHON","")
        
        fields = ['QAQC_CODE']

        edit = arcpy.da.Editor(input_gdb)
        edit.startEditing()
        edit.startOperation()
        with arcpy.da.UpdateCursor(wet_poly,fields) as cursor:
            #arcpy.AddMessage("update cursor processing...")
            for row in cursor:

                if row[0] is None:
                    row[0] = "NNNNNN"
                
                row[0] = row[0][:2]+"N"+row[0][3:]
                newCode = row[0][:2]+"N"+row[0][3:]
                #arcpy.AddMessage("newCode = {x3}".format(x3=newCode))
                cursor.updateRow(row)
                #arcpy.AddMessage("updated row")
        edit.stopOperation()
        edit.stopEditing(True)

        fields = ['QAQC_CODE','ACRES']
        #arcpy.AddMessage("begin cursor")
        edit = arcpy.da.Editor(input_gdb)
        edit.startEditing()
        edit.startOperation()
        with arcpy.da.UpdateCursor(wet_poly,fields) as cursor:
            #arcpy.AddMessage("update cursor processing...")
            for row in cursor:
                #arcpy.AddMessage("row[0]: {x1}, row[1]: {x2}".format(x1=row[0],x2=row[1]))
                if row[0] is None:
                    row[0] = "NNNNNN"
                if row[1] < .01:
                    #arcpy.AddMessage("row[1]<.01")
                    row[0] = row[0][:2]+"S"+row[0][3:]
                    newCode = row[0][:2]+"S"+row[0][3:]
                    #arcpy.AddMessage("newCode = {x3}".format(x3=newCode))
                    cursor.updateRow(row)
                    #arcpy.AddMessage("updated row")

        edit.stopOperation()
        edit.stopEditing(True)
    
        return

class overlappingWetlands_poly(object):
    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "Overlapping Wetlands - Polygonal"
        self.description = "This tool identifies overlapping wetland polygons, and changes character 6 of QAQC_Code = 'O'. \n Polygonal wetland features in the National Wetlands Inventory are not allowed to overlap, should be reviewed, and correct. The tool exports a new feature class named 'Overlapping Wetlands' to the user's geodatabase to help identify the problematic polygons."
         

        self.canRunInBackground = False
        self.category = "Individual Polygonal Tools"

    
    def getParameterInfo(self):
        """Define parameter definitions"""
        # params [0]
        input_gdb = arcpy.Parameter(name = "input_gdb",
                                   displayName = "Input Geodatabase",
                                   datatype = "DEWorkspace",
                                   parameterType = "Required", #Required|Optional|Derived
                                   direction = "Input", #Input|Output
                                   )

        work_area = arcpy.Parameter(name = "work_area",
                                     displayName = "Work Area",
                                     datatype = "String",
                                     parameterType = "Required", #Required|Optional|Derived
                                     direction = "Input", #Input|Output
                                     )
        work_area.value = "CONUS"
        work_area.filter.type = "ValueList"
        work_area.filter.list = ["CONUS","AK","HI","PRVI","PacTrust"]

        #params = None
        
        return [input_gdb, work_area]

    def isLicensed(self):
        """Set whether tool is licensed to execute."""
        return True

    def updateParameters(self, parameters):
        """Modify the values and properties of parameters before internal
        validation is performed.  This method is called whenever a parameter
        has been changed."""
        return

    def updateMessages(self, parameters):
        """Modify the messages created by internal validation for each tool
        parameter.  This method is called after internal validation."""
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        wet_projects = os.path.join(input_gdb,"{work_area}_projects".format(work_area = work_area),"{work_area}_wet_projects".format(work_area=work_area))
        if not arcpy.Exists(wet_poly):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_poly.".format(work_area=work_area))
        if not arcpy.Exists(wet_projects):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_projects.".format(work_area=work_area))

        
        return

    def execute(self, parameters, messages):
        """The source code of the tool."""

        arcpy.AddMessage("Overlapping Wetlands Version 1.4.3")
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))

        arcpy.management.Compact(input_gdb)
        start_count = arcpy.management.GetCount(wet_poly)
        arcpy.AddMessage("Begin Overlapping Wetlands tool")
        overlappingWetlands_poly.overlappingWetlands(input_gdb,work_area)
        end_count = arcpy.management.GetCount(wet_poly)
        if start_count[0] != end_count[0]:
            arcpy.AddWarning("The number of features in {work_area}_wet_poly changed during the course of this tool.".format(work_area=work_area))

        return

    @staticmethod
    def overlappingWetlands(arg1,arg2):
        input_gdb = str(arg1)
        work_area = str(arg2)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        wet_poly_topology = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wetlands_Topology".format(work_area=work_area))
        wet_projects = os.path.join(input_gdb,"{work_area}_projects".format(work_area = work_area),"{work_area}_wet_projects".format(work_area=work_area))
        qaqc_code = "'NNNNNN'"
        #arcpy.management.Compact(input_gdb)

        arcpy.management.MakeFeatureLayer(wet_poly,"wet_poly_lyr")
        fields = ['QAQC_CODE']

        edit = arcpy.da.Editor(input_gdb)
        edit.startEditing()
        edit.startOperation()
        with arcpy.da.UpdateCursor("wet_poly_lyr",fields) as cursor:
            #arcpy.AddMessage("update cursor processing...")
            for row in cursor:

                if row[0] is None:
                    row[0] = "NNNNNN"
                
                row[0] = row[0][:5]+"N"
                newCode = row[0][:5]+"N"
                #arcpy.AddMessage("newCode = {x3}".format(x3=newCode))
                cursor.updateRow(row)
                #arcpy.AddMessage("updated row")
        edit.stopOperation()
        edit.stopEditing(True)

        #arcpy.AddMessage("{},{},{},{}".format(input_gdb,wet_poly,wet_poly_topology,wet_projects))
        
        #arcpy.CalculateField_management(parameters[0].value,"ACRES","!shape.area@acres!","PYTHON","")

        #validate Topology
        arcpy.ValidateTopology_management(wet_poly_topology)

        #delete line & point error classes
        point_topo = os.path.join(input_gdb,"Overlapping_Wetlands_point")
        line_topo = os.path.join(input_gdb,"Overlapping_Wetlands_line")
        poly_topo = os.path.join(input_gdb,"Overlapping_Wetlands_poly")
        poly_topo2 = os.path.join(input_gdb,"Overlapping_Wetlands")

        if arcpy.Exists(poly_topo2):
            arcpy.management.Delete(poly_topo2)
        if arcpy.Exists(poly_topo):
            arcpy.management.Delete(poly_topo)
            
        #export topology errors
        #arcpy.management.ExportTopologyErrors(wet_poly_topology, os.path.join(input_gdb, "{work_area}_wetlands".format(work_area = work_area)),'Overlapping Wetlands')
        #arcpy.management.ExportTopologyErrors(wet_poly_topology, os.path.join(input_gdb,'Overlapping_Wetlands'))
        #arcpy.management.ExportTopologyErrors(wet_poly_topology, os.path.join(input_gdb, "{work_area}_wetlands".format(work_area = work_area)),'Overlapping_Wetlands')
        arcpy.management.ExportTopologyErrors(wet_poly_topology, input_gdb,'Overlapping_Wetlands')

        #delete line & point error classes
        #point_topo = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"Overlapping_Wetlands_point")
        #line_topo = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"Overlapping_Wetlands_line")
        #poly_topo = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"Overlapping_Wetlands_poly")
        if arcpy.Exists(point_topo):
            #arcpy.AddMessage("deleting point topo errors")
            arcpy.management.Delete(point_topo)
        if arcpy.Exists(line_topo):
            #arcpy.AddMessage("deleting line topo errors")
            arcpy.management.Delete(line_topo)
        if arcpy.Exists(poly_topo):
            arcpy.management.Rename(poly_topo,poly_topo2,"FeatureClass")


        arcpy.management.MakeFeatureLayer(wet_poly,"wet_poly_lyr")
        arcpy.management.SelectLayerByLocation("wet_poly_lyr",'INTERSECT',poly_topo2)

        #arcpy.AddMessage("begin cursor")
        fields = ['QAQC_CODE']

        #with arcpy.da.Editor(input_gdb) as edit:
        edit = arcpy.da.Editor(input_gdb)
        edit.startEditing()
        edit.startOperation()
        with arcpy.da.UpdateCursor("wet_poly_lyr",fields) as cursor:
            #arcpy.AddMessage("update cursor processing...")
            for row in cursor:

                if row[0] is None:
                    row[0] = "NNNNNN"

                row[0] = row[0][:5]+"O"
                newCode = row[0][:5]+"O"
                #arcpy.AddMessage("newCode = {x3}".format(x3=newCode))
                cursor.updateRow(row)
                #arcpy.AddMessage("updated row")
        edit.stopOperation()
        edit.stopEditing(True)
                
        return
    

class wetlandTypeCalc_poly(object):
    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "Wetland Type Calculation - Polygonal"
        self.description = "This tool calculates the 'wetland_type' field based on the wetland code in the 'attribute' field. The 'wetland_type' field provides a general description of the wetland type and is used in the cartographic representation of the different wetland types on the Wetlands Mapper."

        self.canRunInBackground = False
        self.category = "Individual Polygonal Tools"

    
    def getParameterInfo(self):
        """Define parameter definitions"""
        # params [0]
        input_gdb = arcpy.Parameter(name = "input_gdb",
                                   displayName = "Input Geodatabase",
                                   datatype = "DEWorkspace",
                                   parameterType = "Required", #Required|Optional|Derived
                                   direction = "Input", #Input|Output
                                   )

        work_area = arcpy.Parameter(name = "work_area",
                                     displayName = "Work Area",
                                     datatype = "String",
                                     parameterType = "Required", #Required|Optional|Derived
                                     direction = "Input", #Input|Output
                                     )
        work_area.value = "CONUS"
        work_area.filter.type = "ValueList"
        work_area.filter.list = ["CONUS","AK","HI","PRVI","PacTrust"]

        #params = None
        
        return [input_gdb, work_area]

    def isLicensed(self):
        """Set whether tool is licensed to execute."""
        return True

    def updateParameters(self, parameters):
        """Modify the values and properties of parameters before internal
        validation is performed.  This method is called whenever a parameter
        has been changed."""
        
        return

    def updateMessages(self, parameters):
        """Modify the messages created by internal validation for each tool
        parameter.  This method is called after internal validation."""
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        wet_projects = os.path.join(input_gdb,"{work_area}_projects".format(work_area = work_area),"{work_area}_wet_projects".format(work_area=work_area))
        if not arcpy.Exists(wet_poly):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_poly.".format(work_area=work_area))

        return

    def execute(self, parameters, messages):
        """The source code of the tool."""

        arcpy.AddMessage("Wetland Type Calculation Version 1.4.3")
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))

        arcpy.management.Compact(input_gdb)
        start_count = arcpy.management.GetCount(wet_poly)
        arcpy.AddMessage("Begin Wetland Type Calculation tool")
        wetlandTypeCalc_poly.wetlandTypeCalc(input_gdb,work_area)
        end_count = arcpy.management.GetCount(wet_poly)
        if start_count[0] != end_count[0]:
            arcpy.AddWarning("The number of features in {work_area}_wet_poly changed during the course of this tool.".format(work_area=work_area))
        
        return

    @staticmethod
    def wetlandTypeCalc(arg1,arg2):

        input_gdb = str(arg1)
        work_area = str(arg2)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))

        fields = ['ATTRIBUTE','WETLAND_TYPE']
        edit = arcpy.da.Editor(input_gdb)
        edit.startEditing()
        edit.startOperation()
        with arcpy.da.UpdateCursor(wet_poly,fields) as cursor:
            #arcpy.AddMessage("update cursor processing...")
            for row in cursor:
                #arcpy.AddMessage("row[0]: {x1}, row[1]: {x2}".format(x1=row[0],x2=row[1]))
                #arcpy.AddMessage("length: {x3}".format(x3=len(row[0])))
                if row[0] is None:
                    row[1] = "NA"
                elif len(row[0])<2:
                    row[1] = "NA"
                elif row[0] == "Pf":
                    row[1] = "Other"
                elif row[0][:2] == "Rp":
                    row[1] = "Riparian"
                elif len(row[0]) < 3:
                    row[1] = "NA"
                elif row[0][:3] == "PFO":
                    row[1] = "Freshwater Forested/Shrub Wetland"
                elif row[0][:3] == "PSS":
                    row[1] = "Freshwater Forested/Shrub Wetland"
                elif row[0][:3] == "PEM":
                    row[1] = "Freshwater Emergent Wetland"
                elif row[0][:3] == "PUB":
                    row[1] = "Freshwater Pond"
                elif row[0][:3] == "PAB":
                    row[1] = "Freshwater Pond"
                elif row[0][:3] == "PUS":
                    row[1] = "Freshwater Pond"
                elif row[0][:2] == "E2":
                    row[1] = "Estuarine and Marine Wetland"
                elif row[0][:2] == "M2":
                    row[1] = "Estuarine and Marine Wetland"
                elif row[0][:2] == "E1":
                    row[1] = "Estuarine and Marine Deepwater"
                elif row[0][:2] == "M1":
                    row[1] = "Estuarine and Marine Deepwater"
                elif row[0][:1] == "R":
                    row[1] = "Riverine"
                elif row[0][:1] == "L":
                    row[1] = "Lake"
                else:
                    row[1] = "Other"

                cursor.updateRow(row)
                #arcpy.AddMessage("updated row")
        edit.stopOperation()
        edit.stopEditing(True)
                                        
        return
    
class lakeAndPondSize(object):
    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "Lake and Pond Size - Polygonal"
        self.description = "Identifies Lakes that are less than 20 acres in size and Ponds that are greater or equal to 20 acres in size. It changes the fifth character of QAQC_Code = 'L' for small lakes or 'P' for large ponds.These may or may not be errors and can be justified based on water depth of the identified waterbody or small lake portions on the edge of the mapping project area. Comments can be added to the ‘comments’field of the QAQC_Summary table for those wetland features flagged that are valid based on depth requirements outlined in the wetlands mapping standards. "
         

        self.canRunInBackground = False
        self.category = "Individual Polygonal Tools"

    
    def getParameterInfo(self):
        """Define parameter definitions"""
        # params [0]
        input_gdb = arcpy.Parameter(name = "input_gdb",
                                   displayName = "Input Geodatabase",
                                   datatype = "DEWorkspace",
                                   parameterType = "Required", #Required|Optional|Derived
                                   direction = "Input", #Input|Output
                                   )

        work_area = arcpy.Parameter(name = "work_area",
                                     displayName = "Work Area",
                                     datatype = "String",
                                     parameterType = "Required", #Required|Optional|Derived
                                     direction = "Input", #Input|Output
                                     )
        work_area.value = "CONUS"
        work_area.filter.type = "ValueList"
        work_area.filter.list = ["CONUS","AK","HI","PRVI","PacTrust"]

        intermediate_data = arcpy.Parameter(name = "intermediate_data_type",
                                            displayName = "Temporary Data Type",
                                            datatype = "GPBoolean",
                                            parameterType = "Required",
                                            direction = "Input"
                                            )

        intermediate_data.value = False
        #params = None
        
        return [input_gdb, work_area,intermediate_data]

    def isLicensed(self):
        """Set whether tool is licensed to execute."""
        return True

    def updateParameters(self, parameters):
        """Modify the values and properties of parameters before internal
        validation is performed.  This method is called whenever a parameter
        has been changed."""
        
        return

    def updateMessages(self, parameters):
        """Modify the messages created by internal validation for each tool
        parameter.  This method is called after internal validation."""
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        wet_projects = os.path.join(input_gdb,"{work_area}_projects".format(work_area = work_area),"{work_area}_wet_projects".format(work_area=work_area))
        if not arcpy.Exists(wet_poly):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_poly.".format(work_area=work_area))
        
            
        return

    def execute(self, parameters, messages):
        """The source code of the tool."""
        
        arcpy.AddMessage("Lake and Pond Size Version 1.4.3")
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        start_count = arcpy.management.GetCount(wet_poly)
        arcpy.management.Compact(input_gdb)
        arcpy.AddMessage("Begin Lake and Pond size tool")
        lakeAndPondSize.lakeAndPondSize(input_gdb,work_area,parameters[2].value)
        end_count = arcpy.management.GetCount(wet_poly)

        if start_count[0] != end_count[0]:
            arcpy.AddWarning("The number of features in {work_area}_wet_poly changed during the course of this tool.".format(work_area=work_area))
        
        return

    @staticmethod
    def lakeAndPondSize(arg1,arg2,arg3):
        input_gdb = str(arg1)
        work_area = str(arg2)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        qaqc_code = "'NNNNNN'"

        #arcpy.management.Compact(input_gdb)

        arcpy.CalculateField_management(wet_poly,"ACRES","!shape.area@acres!","PYTHON","")

        arcpy.management.MakeFeatureLayer(wet_poly,"wet_poly_lyr")
        fields = ['QAQC_CODE']

        edit = arcpy.da.Editor(input_gdb)
        edit.startEditing()
        edit.startOperation()
        with arcpy.da.UpdateCursor("wet_poly_lyr",fields) as cursor:
            #arcpy.AddMessage("update cursor processing...")
            for row in cursor:

                if row[0] is None:
                    row[0] = "NNNNNN"
                
                row[0] = row[0][:4]+"N"+row[0][5:]
                newCode = row[0][:4]+"N"+row[0][5:]
                #arcpy.AddMessage("newCode = {x3}".format(x3=newCode))
                cursor.updateRow(row)
                #arcpy.AddMessage("updated row")
        edit.stopOperation()
        edit.stopEditing(True)

        if arg3 == True:
            pondSelect = r"memory\selectPonds"
            lakeSelect = r"memory\selectLakes"
            pondDissolve = r"memory\SelectPondDissolve"
            lakeDissolve = r"memory\SelectLakeDissolve"
        else:
            pondSelect = os.path.join(input_gdb,"selectPonds")
            lakeSelect = os.path.join(input_gdb,"selectLakes")
            pondDissolve = os.path.join(input_gdb,"SelectPondDissolve")
            lakeDissolve = os.path.join(input_gdb,"SelectLakeDissolve")

        #pondStatement = "\"ATTRIBUTE\" LIKE 'PU%' OR \"ATTRIBUTE\" LIKE 'PA%'"
        pondStatement = "(\"ATTRIBUTE\" LIKE 'PU%' OR \"ATTRIBUTE\" LIKE 'PA%') AND (\"ATTRIBUTE\" NOT LIKE '%EM%' AND \"ATTRIBUTE\" NOT LIKE '%SS%' AND \"ATTRIBUTE\" NOT LIKE '%FO%')"

        lakeStatement = "\"ATTRIBUTE\" LIKE 'L%'"
        arcpy.analysis.Select(wet_poly, pondSelect, pondStatement)
        arcpy.analysis.Select(wet_poly, lakeSelect, lakeStatement)
        
        arcpy.management.MakeFeatureLayer(wet_poly,"wet_poly_lyr")
        
        arcpy.management.CalculateField(pondSelect,"QAQC_CODE","'NNNNNN'")
        arcpy.CalculateField_management(lakeSelect,"QAQC_CODE","'NNNNNN'")

        dissolveField = ['QAQC_CODE']
        statsField = [["ACRES","SUM"]]
        arcpy.management.Dissolve(pondSelect,pondDissolve,dissolveField,statsField,"SINGLE_PART")
        arcpy.management.Dissolve(lakeSelect,lakeDissolve,dissolveField,statsField,"SINGLE_PART")

        arcpy.CalculateField_management(pondDissolve,"SUM_ACRES","!shape.area@acres!","PYTHON","")
        arcpy.CalculateField_management(lakeDissolve,"SUM_ACRES","!shape.area@acres!","PYTHON","")
        
        arcpy.management.MakeFeatureLayer(pondDissolve,"pondDissolvelyr")
        arcpy.management.MakeFeatureLayer(lakeDissolve,"lakeDissolvelyr")

        arcpy.management.SelectLayerByAttribute("pondDissolvelyr","NEW_SELECTION",'"SUM_ACRES" >= 20')
        arcpy.management.SelectLayerByLocation("pondDissolvelyr",'INTERSECT',"lakeDissolvelyr",'',"ADD_TO_SELECTION")
        arcpy.management.SelectLayerByAttribute("lakeDissolvelyr","NEW_SELECTION",'"SUM_ACRES" < 20')

        arcpy.management.SelectLayerByLocation("wet_poly_lyr",'WITHIN',"pondDissolvelyr")
        #arcpy.AddMessage("begin pond cursor")
        fields = ['QAQC_CODE']

        edit = arcpy.da.Editor(input_gdb)
        edit.startEditing()
        edit.startOperation()
        with arcpy.da.UpdateCursor("wet_poly_lyr",fields) as cursor:
            #arcpy.AddMessage("update cursor processing...")
            for row in cursor:

                if row[0] is None:
                    row[0] = "NNNNNN"

                row[0] = row[0][:4]+"P"+row[0][5:]
                newCode = row[0][:4]+"P"+row[0][5:]
                #arcpy.AddMessage("newCode = {x3}".format(x3=newCode))
                cursor.updateRow(row)
                #arcpy.AddMessage("updated row")
        edit.stopOperation()
        edit.stopEditing(True)

        arcpy.management.SelectLayerByLocation("wet_poly_lyr",'WITHIN',"lakeDissolvelyr","","NEW_SELECTION")
        #arcpy.AddMessage("begin lake cursor")

        edit = arcpy.da.Editor(input_gdb)
        edit.startEditing()
        edit.startOperation()
        with arcpy.da.UpdateCursor("wet_poly_lyr",fields) as cursor:
            #arcpy.AddMessage("update cursor processing...")
            for row in cursor:

                if row[0] is None:
                    row[0] = "NNNNNN"
                
                row[0] = row[0][:4]+"L"+row[0][5:]
                newCode = row[0][:4]+"L"+row[0][5:]
                #arcpy.AddMessage("newCode = {x3}".format(x3=newCode))
                cursor.updateRow(row)
                #arcpy.AddMessage("updated row")
        edit.stopOperation()
        edit.stopEditing(True)

        if arg3 == False:
            fc_lakePond_temp_list = [pondDissolve,lakeDissolve,pondSelect,lakeSelect]
            for fc_lPs in fc_lakePond_temp_list:
                if arcpy.Exists(fc_lPs):
                    #arcpy.AddMessage("deleting temporary lake and pond layers")
                    #arcpy.AddMessage(fc_lPs)
                    arcpy.management.Delete(fc_lPs)
        
        return
        
class adjacentWetlands(object):
    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "Adjacent Wetlands - Polygonal"
        self.description = "Identifies wetland polygons that are adjacent to other wetland polygons with the same 'attribute'. Changes character 2 of QAQC_Code = 'A'."
         

        self.canRunInBackground = False
        self.category = "Individual Polygonal Tools"

    
    def getParameterInfo(self):
        """Define parameter definitions"""
        # params [0]
        input_gdb = arcpy.Parameter(name = "input_gdb",
                                   displayName = "Input Geodatabase",
                                   datatype = "DEWorkspace",
                                   parameterType = "Required", #Required|Optional|Derived
                                   direction = "Input", #Input|Output
                                   )

        work_area = arcpy.Parameter(name = "work_area",
                                     displayName = "Work Area",
                                     datatype = "String",
                                     parameterType = "Required", #Required|Optional|Derived
                                     direction = "Input", #Input|Output
                                     )
        work_area.value = "CONUS"
        work_area.filter.type = "ValueList"
        work_area.filter.list = ["CONUS","AK","HI","PRVI","PacTrust"]

        intermediate_data = arcpy.Parameter(name = "intermediate_data_type",
                                            displayName = "Temporary Data Type",
                                            datatype = "GPBoolean",
                                            parameterType = "Required",
                                            direction = "Input"
                                            )

        intermediate_data.value = False

        #params = None
        
        return [input_gdb, work_area,intermediate_data]

    def isLicensed(self):
        """Set whether tool is licensed to execute."""
        return True

    def updateParameters(self, parameters):
        """Modify the values and properties of parameters before internal
        validation is performed.  This method is called whenever a parameter
        has been changed."""
        
        return

    def updateMessages(self, parameters):
        """Modify the messages created by internal validation for each tool
        parameter.  This method is called after internal validation."""
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        wet_projects = os.path.join(input_gdb,"{work_area}_projects".format(work_area = work_area),"{work_area}_wet_projects".format(work_area=work_area))
        if not arcpy.Exists(wet_poly):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_poly.".format(work_area=work_area))

        return

    def execute(self, parameters, messages):
        """The source code of the tool."""

        arcpy.AddMessage("Adjacent Wetlands Version 1.4.3")
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        #qaqc_code = "'NNNNNN'"

        arcpy.management.Compact(input_gdb)
        start_count = arcpy.management.GetCount(wet_poly)
        arcpy.AddMessage("Begin Adjacent Wetlands tool")
        adjacentWetlands.adjacentWetlands(input_gdb,work_area,parameters[2].value)
        end_count = arcpy.management.GetCount(wet_poly)
        if start_count[0] != end_count[0]:
            arcpy.AddWarning("The number of features in {work_area}_wet_poly changed during the course of this tool.".format(work_area=work_area))
        
        return

    @staticmethod
    def adjacentWetlands(arg1,arg2,arg3):
        arcpy.env.maintainCurveSegments = True
        input_gdb = str(arg1)
        work_area = str(arg2)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))

        if arg3 == True:
            adjacentDissolve = r"memory\dissolveAdjacent"
        else:
            adjacentDissolve = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"adjacentDissolve")

        dissolveField = ['ATTRIBUTE']
        
        arcpy.management.Dissolve(wet_poly,adjacentDissolve,dissolveField, "", "SINGLE_PART")
        #arcpy.analysis.PairwiseDissolve(wet_poly,adjacentDissolve,dissolveField, "", "SINGLE_PART")
        arcpy.management.MakeFeatureLayer(adjacentDissolve,"adj_diss_lyr")
        arcpy.management.MakeFeatureLayer(wet_poly,"wet_poly_lyr")
        
        #arcpy.AddMessage("begin Adjacent Reset cursor")
        fields = ['QAQC_CODE']

        edit = arcpy.da.Editor(input_gdb)
        edit.startEditing()
        edit.startOperation()
        with arcpy.da.UpdateCursor("wet_poly_lyr",fields) as cursor:
            #arcpy.AddMessage("update cursor processing...")
            for row in cursor:

                if row[0] is None:
                    row[0] = "NNNNNN"
                
                row[0] = row[0][:1]+"N"+row[0][2:]
                newCode = row[0][:1]+"N"+row[0][2:]
                #arcpy.AddMessage("newCode = {x3}".format(x3=newCode))
                cursor.updateRow(row)
                #arcpy.AddMessage("updated row")
        edit.stopOperation()
        edit.stopEditing(True)
        
        arcpy.management.SelectLayerByLocation("wet_poly_lyr",'ARE_IDENTICAL_TO',"adj_diss_lyr","","","INVERT")

        #begin changing adjacent features' codes to *A****
        edit = arcpy.da.Editor(input_gdb)
        edit.startEditing()
        edit.startOperation()
        with arcpy.da.UpdateCursor("wet_poly_lyr",fields) as cursor:
            #arcpy.AddMessage("update cursor processing...")
            for row in cursor:

                if row[0] is None:
                    row[0] = "NNNNNN"
                
                row[0] = row[0][:1]+"A"+row[0][2:]
                newCode = row[0][:1]+"A"+row[0][2:]
                #arcpy.AddMessage("newCode = {x3}".format(x3=newCode))
                cursor.updateRow(row)
                #arcpy.AddMessage("updated row")
        edit.stopOperation()
        edit.stopEditing(True)
        if arcpy.Exists(adjacentDissolve):
            arcpy.management.Delete(adjacentDissolve)

        
        mp_lyr = arcpy.management.MakeFeatureLayer(wet_poly) 
        mpSelect = os.path.join(input_gdb,"Multipart_Wetland_Features")
        fields2 = ['OBJECTID','SHAPE@',"QAQC_CODE"]
        multipartList = []
        #arcpy.AddMessage("start mp cursor")
        edit.startEditing()
        edit.startOperation()
        with arcpy.da.UpdateCursor(mp_lyr,fields2) as cursor2:
            for row in cursor2:
                if row[1] is None:
                    cursor2.deleteRow()
                elif row[1].partCount > 1:
                    multipartList.append(row[0])
                    row[2] = row[2][:1]+"A"+row[2][2:]
                    cursor2.updateRow(row)
        edit.stopOperation()
        edit.stopEditing(True)
        
        #arcpy.AddMessage(multipartList)
        #mp_lyr[0].setSelectionSet(multipartList,'NEW')
        #arcpy.analysis.Select(mp_lyr[0], mpSelect)
        if len(multipartList) > 0:
            if len(multipartList) == 1:
                qry = """OBJECTID IN {0})""".format(str(tuple(multipartList))[:-2])
            else:
                qry = """OBJECTID IN {0}""".format(str(tuple(multipartList)))
            #arcpy.AddMessage(qry)
            arcpy.conversion.ExportFeatures(mp_lyr,mpSelect,qry)
                              
        return

class QAQCSummary_poly(object):
    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "QAQC Summary - Polygonal"
        self.description = "Summarizes the QAQC_CODE field into a 'QAQC_Summary' table in your wetlands file geodatabase. It also describes each error type and records who conducted the verification and when the verification was run. Comments can be added to the ‘comments’ field of the QAQC_Summary table to justify specific types of errors. These comments will only be saved if the ‘Append to History Table’ box is checked prior to running the All_QAQC_Checks model. "
         

        self.canRunInBackground = False
        self.category = "Individual Polygonal Tools"

    
    def getParameterInfo(self):
        """Define parameter definitions"""
        # params [0]
        input_gdb = arcpy.Parameter(name = "input_gdb",
                                   displayName = "Input Geodatabase",
                                   datatype = "DEWorkspace",
                                   parameterType = "Required", #Required|Optional|Derived
                                   direction = "Input", #Input|Output
                                   )

        work_area = arcpy.Parameter(name = "work_area",
                                     displayName = "Work Area",
                                     datatype = "String",
                                     parameterType = "Required", #Required|Optional|Derived
                                     direction = "Input", #Input|Output
                                     )
        work_area.value = "CONUS"
        work_area.filter.type = "ValueList"
        work_area.filter.list = ["CONUS","AK","HI","PRVI","PacTrust"]

        verifier_name = arcpy.Parameter(name = "verifier_name",
                                        displayName = "Verified By:",
                                        datatype = "String",
                                        parameterType = "Required",
                                        direction = "Input",
                                        )

        save_summary_table = arcpy.Parameter(name = "save_summary_table",
                                             displayName = "Save Summary Table?",
                                             datatype = "GPBoolean",
                                             parameterType = "Required",
                                             direction = "Input",
                                             )
        save_summary_table.value = True
        #params = None
        
        return [input_gdb, work_area,verifier_name,save_summary_table]

    def isLicensed(self):
        """Set whether tool is licensed to execute."""
        return True

    def updateParameters(self, parameters):
        """Modify the values and properties of parameters before internal
        validation is performed.  This method is called whenever a parameter
        has been changed."""

        return

    def updateMessages(self, parameters):
        """Modify the messages created by internal validation for each tool
        parameter.  This method is called after internal validation."""
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        wet_projects = os.path.join(input_gdb,"{work_area}_projects".format(work_area = work_area),"{work_area}_wet_projects".format(work_area=work_area))
        if not arcpy.Exists(wet_poly):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_poly.".format(work_area=work_area))
        

        return

    def execute(self, parameters, messages):
        """The source code of the tool."""

        arcpy.AddMessage("QAQC Summary Version 1.4.3")
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        verifier_name = str(parameters[2].value)
        save_summary_table = str(parameters[3].value)
        #wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        
        arcpy.management.Compact(input_gdb)

        arcpy.AddMessage("Begin QAQC Summary tool")
        QAQCSummary_poly.QAQCSummary_tool(input_gdb,work_area,verifier_name,save_summary_table)
                        
        return

    @staticmethod
    def QAQCSummary_tool(arg1,arg2,arg3,arg4):

        input_gdb = str(arg1)
        work_area = str(arg2)
        verifier_name = str(arg3)
        save_summary_table = arg4
        #arcpy.AddMessage("arg4: {x1},type(arg4): {x2}".format(x1=arg4,x2=type(arg4)))
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        wet_line = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_line".format(work_area=work_area))
        summaryTable = os.path.join(input_gdb,"QAQC_Summary")
        #arcpy.management.Compact(input_gdb)
        tempLinearTable = r"memory/tempLinearTable"
        arcpy.analysis.Frequency(wet_poly,summaryTable,["QAQC_CODE"])
        if arcpy.Exists(wet_line):
            arcpy.analysis.Frequency(wet_line,tempLinearTable,["QAQC_CODE"])
            arcpy.management.Append(tempLinearTable,summaryTable,"TEST")
        arcpy.management.AddField(summaryTable,"ERROR_TYPE","TEXT","","",255)
        arcpy.management.AddField(summaryTable,"VERIFIED_BY","TEXT","","",255)
        arcpy.management.AddField(summaryTable,"VERIFICATION_DATE","DATE")
        arcpy.management.AddField(summaryTable,"VERIFICATION_COMMENTS","TEXT","","",255)
        
        
        #arcpy.AddMessage("begin cursor")
        fields = ['QAQC_CODE','ERROR_TYPE']
        

        edit = arcpy.da.Editor(input_gdb)
        edit.startEditing()
        edit.startOperation()
        with arcpy.da.UpdateCursor(summaryTable,fields) as cursor:
            #arcpy.AddMessage("update cursor processing...")
            for row in cursor:
                newList = []

                if row[0] == None:
                    row[1] = "Null Value in QAQC_CODE"
                elif len(row[0]) == 6:
                    if row[0] == "NNNNNN":
                        row[1] = "Polygons Passed Verification"
                    else:
                        printDict = {
                            'C': 'Bad Polygonal Code',
                            'A': 'Adjacent Polygon',
                            'S': 'Sliver Polygonal Wetland',
                            'U': 'Sliver Polygonal Upland',
                            'G': 'Gap Polygonal Wetland',
                            'L': 'Small Lake',
                            'P': 'Large Pond',
                            'O': 'Overlapping Polygonal Wetland',
                            '': 'Null Values',
                            ',': ', '
                            }
                        #newList = []
                        #newListString = ''
                        for i,v in enumerate(row[0]):
                            #arcpy.AddMessage(v)
                            #arcpy.AddMessage(type(newList))
                            if v != 'N':
                                newList.append(v)
                                
                            newListComma = ','.join(newList)
                            #arcpy.AddMessage('newListComma: {x}'.format(x=newListComma))
                            
                            printString = ''
                            for char in newListComma:
                                nLC = str(printDict[char])
                                printString += nLC

                        #arcpy.AddMessage('printString: {y}'.format(y=printString))
                        row[1] = printString

                elif len(row[0]) == 5:    
                    if row[0] == "NNNNN":
                        row[1] = "Linears Passed Verification"
                    else:
                        printDict = {
                            'C': 'Bad Linear Code',
                            'W': 'Linear Overlapping Polygonal Wetland',
                            'R': 'Connecting Linear Wetland',
                            'S': 'Sliver Linear Wetland',
                            'O': 'Overlapping Linear Wetland',
                            'G': 'Gap Linear Wetland',
                            ',': ', '
                            }
                        #newList = []
                        #newListString = ''
                        for i,v in enumerate(row[0]):
                            #arcpy.AddMessage(v)
                            #arcpy.AddMessage(type(newList))
                            if v != 'N':
                                newList.append(v)
                                
                            newListComma = ','.join(newList)
                            #arcpy.AddMessage('newListComma: {x}'.format(x=newListComma))
                            
                            printString = ''
                            for char in newListComma:
                                nLC = str(printDict[char])
                                printString += nLC

                            #arcpy.AddMessage('printString: {y}'.format(y=printString))
                            row[1] = printString
                    
                cursor.updateRow(row)
                #arcpy.AddMessage("updated row")
                
        edit.stopOperation()
        edit.stopEditing(True)

        arcpy.CalculateField_management(summaryTable,"VERIFIED_BY","'"+verifier_name+"'","PYTHON","")
        #date_value = datetime.datetime.now()
        arcpy.CalculateField_management(summaryTable,"VERIFICATION_DATE","datetime.datetime.now()","PYTHON","")

        history_table = os.path.join(input_gdb,"QAQC_History")
        if arcpy.Exists(history_table):
            run_list = []
            with arcpy.da.SearchCursor(history_table,'RUN') as cursor:
                for row in cursor:
                    run_list.append(row[0])
            run_max = max(run_list)
            #arcpy.AddMessage("run_max: {x1}".format(x1=run_max))
        
            #arcpy.AddMessage("history table exists")

            arcpy.AddField_management(summaryTable,"RUN","SHORT")
            arcpy.CalculateField_management(summaryTable,"RUN",run_max+1)
            arcpy.management.Append(summaryTable,history_table,"TEST")
        else:
            arcpy.AddField_management(summaryTable,"RUN","SHORT")
            arcpy.CalculateField_management(summaryTable,"RUN",1)
            #arcpy.CopyFeatures_management(summaryTable,history_table)
            arcpy.conversion.TableToTable(summaryTable,input_gdb,'QAQC_History')


        return

class sliverUplands_poly(object):
    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "Sliver Uplands - Polygonal"
        self.description = "Identifies upland islands or holes in wetlands that are less than 0.01 acres. These may be actual upland features but are identified as errors as they are typically errors in wetland mapping. The model changes the fourth character of QAQC_Code = 'U', in all wetland polygons adjacent to the upland sliver. The sliver upland polygons are stored as a new feature class ‘Sliver Uplands’ in your wetlands file geodatabase to assist in locating these small geographic features for review. This tool requires that a 'CONUS_wet_projects' has a feature(s) that defines the wetland mapping project and completely covers all features in the 'CONUS_wet_poly' feature class. NOTE: This tool is a computationally intensive process and may fail on extremely large geographic areas. "
         

        self.canRunInBackground = False
        self.category = "Individual Polygonal Tools"

    
    def getParameterInfo(self):
        """Define parameter definitions"""
        # params [0]
        input_gdb = arcpy.Parameter(name = "input_gdb",
                                   displayName = "Input Geodatabase",
                                   datatype = "DEWorkspace",
                                   parameterType = "Required", #Required|Optional|Derived
                                   direction = "Input", #Input|Output
                                   )

        work_area = arcpy.Parameter(name = "work_area",
                                     displayName = "Work Area",
                                     datatype = "String",
                                     parameterType = "Required", #Required|Optional|Derived
                                     direction = "Input", #Input|Output
                                     )
        work_area.value = "CONUS"
        work_area.filter.type = "ValueList"
        work_area.filter.list = ["CONUS","AK","HI","PRVI","PacTrust"]

        intermediate_data = arcpy.Parameter(name = "intermediate_data_type",
                                            displayName = "Temporary Data Type",
                                            datatype = "GPBoolean",
                                            parameterType = "Required",
                                            direction = "Input"
                                            )

        intermediate_data.value = False

        project_division = arcpy.Parameter(name = "project_division_number",
                                           displayName = "Number of Processing Divisions",
                                           datatype = "GPLong",
                                           parameterType = "Required",
                                           direction = "Input")

        project_division.value = 1
        project_division.filter.type = "ValueList"
        project_division.filter.list = [1,4,9,16,25]

        #params = None
        
        return [input_gdb, work_area,intermediate_data,project_division]
    
    def isLicensed(self):
        """Set whether tool is licensed to execute."""
        return True

    def updateParameters(self, parameters):
        """Modify the values and properties of parameters before internal
        validation is performed.  This method is called whenever a parameter
        has been changed."""

        return

    def updateMessages(self, parameters):
        """Modify the messages created by internal validation for each tool
        parameter.  This method is called after internal validation."""

        
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        wet_projects = os.path.join(input_gdb,"{work_area}_projects".format(work_area = work_area),"{work_area}_wet_projects".format(work_area=work_area))
        if not arcpy.Exists(wet_poly):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_poly.".format(work_area=work_area))
        if not arcpy.Exists(wet_projects):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_projects.".format(work_area=work_area))
        

        return

    def execute(self, parameters, messages):
        """The source code of the tool."""

        arcpy.AddMessage("Sliver Uplands Version 1.4.3")
        input_gdb = str(parameters[0].value)
        arcpy.AddMessage(input_gdb.count(".gdb"))
        work_area = str(parameters[1].value)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        wet_projects = os.path.join(input_gdb,"{work_area}_projects".format(work_area = work_area),"{work_area}_wet_projects".format(work_area=work_area))

        #qaqc_code = "'NNNNNN'"
        arcpy.management.Compact(input_gdb)

        start_count = arcpy.management.GetCount(wet_poly)
        arcpy.AddMessage("Begin Sliver Uplands tool")
        sliverUplands_poly.sliverUplands(input_gdb,work_area,parameters[2].value,parameters[3].value)
        end_count = arcpy.management.GetCount(wet_poly)
        if start_count[0] != end_count[0]:
            arcpy.AddWarning("The number of features in {work_area}_wet_poly changed during the course of this tool.".format(work_area=work_area))

        return

    @staticmethod
    def sliverUplands(arg1,arg2,arg3,arg4):
        arcpy.env.maintainCurveSegments = True
        input_gdb = str(arg1)
        work_area = str(arg2)
        
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        rip_poly = os.path.join(input_gdb,"{work_area}_riparian".format(work_area = work_area),"{work_area}_rip_poly".format(work_area=work_area))
        wet_projects = os.path.join(input_gdb,"{work_area}_projects".format(work_area = work_area),"{work_area}_wet_projects".format(work_area=work_area))
        sliver_uplands = os.path.join(input_gdb,"Sliver_Uplands")

        #arcpy.AddMessage(f"Sliver Uplands {sliver_uplands} already exists?: {arcpy.Exists(sliver_uplands)}")

        qaqc_code = "'NNNNNN'"
        #arcpy.management.Compact(input_gdb)

        arcpy.management.MakeFeatureLayer(wet_poly,"wet_poly_lyr")
        arcpy.CalculateField_management(wet_poly,"ACRES","!shape.area@acres!","PYTHON","")

        #this catches the donut temporary layers
        donut_list = []
        
        ### This section resets the QAQC Code value to N
        fields = ['QAQC_CODE']
        #arcpy.AddMessage("begin cursor")
        edit = arcpy.da.Editor(input_gdb)
        edit.startEditing()
        edit.startOperation()
        with arcpy.da.UpdateCursor(wet_poly,fields) as cursor:
            #arcpy.AddMessage("update cursor processing...")
            for row in cursor:
                if row[0] is None:
                    row[0] = "NNNNNN"
                    
                row[0] = row[0][:3]+"N"+row[0][4:]
                cursor.updateRow(row)
                #arcpy.AddMessage("updated row")

        edit.stopOperation()
        edit.stopEditing(True)

        ### This section checks for slivers between wetland features by dissolving them and then comparing the partCount to the boundary partCount

        if arg4 >1:
            proj_div = math.sqrt(arg4)
            wet_projects4 = wet_projects+"_{a}".format(a=str(int(proj_div)))
            wet_projects16 = wet_projects+"_{a}".format(a=str(arg4))
            arcpy.management.SubdividePolygon(wet_projects,wet_projects4, 'NUMBER_OF_EQUAL_PARTS',proj_div,'','',0, 'STRIPS')
            arcpy.management.SubdividePolygon(wet_projects4,wet_projects16, 'NUMBER_OF_EQUAL_PARTS',proj_div,'','',0, 'STACKED_BLOCKS')
        else:
            wet_projects16 = wet_projects
            
        
        if arg3 == True:
            tempDissolve = r"memory\tempDissolve"
            dissolveDonuts = r"memory\dissolveDonuts"
            dissolveDonuts_union = r"memory\dissolveDonuts_union"
            slivers_temp = r"memory\slivers_temp"
        else:
            tempDissolve = os.path.join(input_gdb,"tempDissolve")
            dissolveDonuts = os.path.join(input_gdb,"dissolveDonuts")
            dissolveDonuts_union = os.path.join(input_gdb,"dissolveDonuts_union")
            slivers_temp = os.path.join(input_gdb,"slivers_temp")

        fields = ["OBJECTID","SHAPE@"]
        tempDissolve_fc_list = []
        with arcpy.da.SearchCursor(wet_projects16,fields) as cursor:
            for row in cursor:
                tempDissolve_x = tempDissolve+str(row[0])
                tempDissolve_fc_list.append(tempDissolve_x)
                arcpy.management.SelectLayerByLocation("wet_poly_lyr",'INTERSECT',row[1],'200 Meters')
                arcpy.management.Dissolve("wet_poly_lyr",tempDissolve_x,'','','SINGLE_PART')

                arcpy.management.AddField(tempDissolve_x,"PART_COUNT","DOUBLE")

                f2 = ["PART_COUNT","SHAPE@"]
                with arcpy.da.UpdateCursor(tempDissolve_x,f2) as f_cursor:
                    for row in f_cursor:
                        if row[1] is None:
                            f_cursor.deleteRow()
                        else:
                            parts = row[1].partCount
                            boundaries = row[1].boundary().partCount
                            if boundaries > parts:
                                row[0] = 1
                            else:
                                row[0] = 0
                            f_cursor.updateRow(row)

        fields2 = ['PART_COUNT','SHAPE@']

        #arcpy.AddMessage(tempDissolve_fc_list)
        sliver_x_list = []
        fc_count = 1
        for fc in tempDissolve_fc_list:
            dissolveDonuts_x = dissolveDonuts+str(fc_count)
            dissolveDonuts_union_x = dissolveDonuts_union+str(fc_count)
            donut_list.append(dissolveDonuts_x)
            donut_list.append(dissolveDonuts_union_x)
            arcpy.management.MakeFeatureLayer(fc,"tempDissolve_lyr")
            arcpy.management.SelectLayerByAttribute("tempDissolve_lyr","NEW_SELECTION",'"PART_COUNT" > 0')
            arcpy.management.CopyFeatures("tempDissolve_lyr",dissolveDonuts_x)
            arcpy.analysis.Union(dissolveDonuts_x,dissolveDonuts_union_x,"","","NO_GAPS")

            arcpy.management.AddField(dissolveDonuts_union_x,"ACRES","DOUBLE")
            arcpy.CalculateField_management(dissolveDonuts_union_x,"ACRES","!shape.area@acres!","PYTHON","")

            arcpy.management.MakeFeatureLayer(dissolveDonuts_union_x,"Sliver_uplands_lyr")
            arcpy.management.SelectLayerByAttribute("Sliver_uplands_lyr","NEW_SELECTION",'"FID_dissolveDonuts{x}" = -1 AND "ACRES" <= 0.01'.format(x=fc_count))
            arcpy.management.CopyFeatures("Sliver_uplands_lyr",slivers_temp+str(fc_count))
            sliver_x_list.append(slivers_temp+str(fc_count))
            fc_count +=1
                                          
        #arcpy.AddMessage(sliver_x_list)
        sliver_uplands = os.path.join(input_gdb,"Sliver_Uplands")
        
        arcpy.management.Merge(sliver_x_list,sliver_uplands)
        arcpy.management.DeleteField(sliver_uplands,["FID_dissolveDonuts"])
        
        arcpy.management.MakeFeatureLayer(sliver_uplands,"Sliver_uplands_lyr")
        sliv_count = int(arcpy.management.GetCount("Sliver_uplands_lyr")[0])

        if sliv_count == 0:
            arcpy.AddWarning("There were no sliver uplands found in the interior of the project.")
        elif sliv_count > 0:
            arcpy.management.SelectLayerByLocation("wet_poly_lyr","INTERSECT","Sliver_uplands_lyr")

            fields = ['QAQC_CODE']
            #arcpy.AddMessage("begin QAQC Code cursor")
            edit = arcpy.da.Editor(input_gdb)
            edit.startEditing()
            edit.startOperation()
            with arcpy.da.UpdateCursor("wet_poly_lyr",fields) as cursor:
                #arcpy.AddMessage("update cursor processing...")
                for row in cursor:
                    row[0] = row[0][:3]+"U"+row[0][4:]

                    cursor.updateRow(row)
                    #arcpy.AddMessage("updated row")

            edit.stopOperation()
            edit.stopEditing(True)


        ### This section checks for gaps between features and the project boundary. ###
        #arcpy.AddMessage("Beginning the search for gaps along the project boundary.")

        if arg3 == True:
            wet_projects_line = r"memory\projectLine"
            wet_projects_line_5m = r"memory\projects_line_5m"
            wet_projects_line_20m = r"memory\projects_line_20m"
            wet_poly_clipped = r"memory\wet_poly_clipped"
            wet_poly_clipped_20 = r"memory\wet_poly_clipped20m"
            rip_poly_clipped_20 = r"memory\rip_poly_clipped20m"
        else:
            wet_projects_line = os.path.join(input_gdb,"projectLine")
            wet_projects_line_5m = os.path.join(input_gdb,"projects_line_5m")
            wet_projects_line_20m = os.path.join(input_gdb,"projects_line_20m")
            wet_poly_clipped = os.path.join(input_gdb,"wet_poly_clipped")
            wet_poly_clipped_20 = os.path.join(input_gdb,"wet_poly_clipped20m")
            rip_poly_clipped_20 = os.path.join(input_gdb,"rip_poly_clipped20m")
        
        
        
        #wet_projects_line_point01m = os.path.join(input_gdb,"projects_line_point01m")
        #wet_projects_line_point01m = r"memory\projects_line_point01m"
        
        
        #wet_poly_clipped = r"memory\wet_poly_clipped"
        #wetland_gap_check = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"Potential_Boundary_Gaps".format(work_area=work_area))
        wetland_gap_check = os.path.join(input_gdb,"Potential_Boundary_Gaps")
        

        arcpy.management.PolygonToLine(wet_projects,wet_projects_line)
        arcpy.analysis.Buffer(wet_projects_line,wet_projects_line_5m,"5 Meters")
        arcpy.analysis.Buffer(wet_projects_line,wet_projects_line_20m,"20 Meters")
        #arcpy.analysis.Buffer(wet_projects_line,wet_projects_line_point01m,".01 Meters")
        arcpy.analysis.Clip(wet_poly,wet_projects_line_20m,wet_poly_clipped_20)
        if arcpy.Exists(rip_poly):
            arcpy.analysis.Clip(rip_poly,wet_projects_line_20m,rip_poly_clipped_20)
        arcpy.analysis.Clip(wet_poly_clipped_20,wet_projects_line_5m,wet_poly_clipped)
        
        arcpy.management.MakeFeatureLayer(wet_poly_clipped,"wet_poly_clipped_lyr")
        arcpy.management.SelectLayerByLocation("wet_poly_clipped_lyr","WITHIN_A_DISTANCE",wet_projects_line, '5 Meters')
        arcpy.management.SelectLayerByLocation("wet_poly_clipped_lyr","INTERSECT",wet_projects_line,"","REMOVE_FROM_SELECTION")
        arcpy.management.SelectLayerByLocation("wet_poly_lyr","CONTAINS","wet_poly_clipped_lyr")
        #project_erase = os.path.join(input_gdb,"projectErase")
        #arcpy.analysis.PairwiseErase(wet_projects,"wet_poly_lyr",project_erase)
        #arcpy.management.CopyFeatures("wet_poly_lyr",wetland_gap_check)
        
        arcpy.management.CopyFeatures("wet_poly_clipped_lyr",wetland_gap_check)
        #arcpy.analysis.Buffer("wet_poly_lyr",os.path.join(input_gdb,"poly_buffered"),".01 Meters")
        #arcpy.analysis.Intersect([wet_projects_line_point01m,os.path.join(input_gdb,"poly_buffered")],os.path.join(input_gdb,"gap_intersect"))

        fields = ['QAQC_CODE']
        #arcpy.AddMessage("begin QAQC Code cursor for N/G")
        edit = arcpy.da.Editor(input_gdb)
        edit.startEditing()
        edit.startOperation()
        with arcpy.da.UpdateCursor("wet_poly_lyr",fields) as cursor:
            #arcpy.AddMessage("update cursor processing...")
            for row in cursor:
                row[0] = row[0][:3]+"G"+row[0][4:]

                cursor.updateRow(row)
                #arcpy.AddMessage("updated row")

        edit.stopOperation()
        edit.stopEditing(True)

        #now address actual sliver uplands that exist on the project boundary
        
        project_buffer_erased = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"boundary_slivers_erased".format(work_area=work_area))
        project_buffer_exploded = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"boundary_slivers_exploded".format(work_area=work_area))
        project_buffer_erased_small = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"boundary_slivers".format(work_area=work_area))
        project_buffer_erased_small2 = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"boundary_slivers2".format(work_area=work_area))
        project_buffer_clipped = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"boundary_slivers_clipped".format(work_area=work_area))

        boundarySliverBuffer = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"boundarySliverBuffer".format(work_area=work_area))
        boundarySliverBufferDissolve = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"boundarySliverBufferDissolve".format(work_area=work_area))
        boundarySliverBufferDissolveEliminate = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"boundarySliverBufferDissolveEliminate".format(work_area=work_area))
        arcpy.analysis.PairwiseErase(wet_projects_line_5m,wet_projects,boundarySliverBuffer)
        arcpy.management.Append(wet_poly_clipped_20,boundarySliverBuffer,"NO_TEST")
        if arcpy.Exists(rip_poly):
            arcpy.management.Append(rip_poly_clipped_20,boundarySliverBuffer,"NO_TEST")
        arcpy.management.Dissolve(boundarySliverBuffer,boundarySliverBufferDissolve,'','','SINGLE_PART')
        arcpy.management.EliminatePolygonPart(boundarySliverBufferDissolve,boundarySliverBufferDissolveEliminate,condition="AREA",part_area="0.3 Acres",part_area_percent=0,part_option="CONTAINED_ONLY")
        
        arcpy.analysis.PairwiseErase(wet_projects_line_20m,wet_poly_clipped_20,project_buffer_erased)
        arcpy.analysis.PairwiseClip(project_buffer_erased,wet_projects,project_buffer_clipped)
        arcpy.management.MultipartToSinglepart(project_buffer_clipped,project_buffer_exploded)
        arcpy.management.AddField(project_buffer_exploded,"ACRES","DOUBLE")
        arcpy.CalculateField_management(project_buffer_exploded,"ACRES","!shape.area@acres!","PYTHON","")
        arcpy.management.MakeFeatureLayer(project_buffer_exploded,"pbe_lyr")
        arcpy.management.SelectLayerByLocation("pbe_lyr","INTERSECT",wet_projects_line)
        arcpy.management.SelectLayerByLocation("pbe_lyr","COMPLETELY_WITHIN",boundarySliverBufferDissolveEliminate,'',"SUBSET_SELECTION")
        arcpy.management.SelectLayerByAttribute("pbe_lyr","SUBSET_SELECTION",'"ACRES" <= 0.01')
        #arcpy.AddMessage(f"boundary slivers below 0.01 acres: {arcpy.management.GetCount('pbe_lyr')}")

        if int(arcpy.management.GetCount("pbe_lyr")[0]) > 0:
            arcpy.analysis.Select('pbe_lyr',project_buffer_erased_small,'"ACRES" <= 0.01')
            arcpy.management.Append(project_buffer_erased_small,sliver_uplands,"NO_TEST")
                    #Select wet_poly features that intersect the boundary slivers layer
            arcpy.management.SelectLayerByLocation("wet_poly_lyr","INTERSECT",project_buffer_erased_small)

            fields = ['QAQC_CODE']
            #arcpy.AddMessage("begin QAQC Code cursor for boundary slivers")
            edit = arcpy.da.Editor(input_gdb)
            edit.startEditing()
            edit.startOperation()
            with arcpy.da.UpdateCursor("wet_poly_lyr",fields) as cursor:
                #arcpy.AddMessage("update cursor processing...")
                for row in cursor:
                    row[0] = row[0][:3]+"U"+row[0][4:]

                    cursor.updateRow(row)
                    #arcpy.AddMessage("updated row")

            edit.stopOperation()
            edit.stopEditing(True)

        else:
            arcpy.AddWarning("There were no sliver uplands found on the project boundary.")
        #arcpy.analysis.Select('pbe_lyr',project_buffer_erased_small2,'ACRES <= 0.03')
        #arcpy.analysis.Select('pbe_lyr',project_buffer_erased_small,'"ACRES" <= 0.2')
        

        

        fc_sliver_temp_list = [project_buffer_erased,
                               project_buffer_exploded,
                               project_buffer_clipped,
                               dissolveDonuts,
                               dissolveDonuts_union,
                               wet_projects_line,
                               wet_projects_line_5m,
                               wet_projects_line_20m,
                               project_buffer_erased_small,
                               boundarySliverBuffer,
                               boundarySliverBufferDissolve,
                               boundarySliverBufferDissolveEliminate,
                               wet_poly_clipped,
                               wet_poly_clipped_20,
                               rip_poly_clipped_20]

        fc_sliver_temp_list.extend(tempDissolve_fc_list)
        fc_sliver_temp_list.extend(sliver_x_list)
        fc_sliver_temp_list.extend(donut_list)
        #arcpy.AddMessage("deleting temporary boundary sliver layers")
        
        
        for fc_slivers in fc_sliver_temp_list:
            if arcpy.Exists(fc_slivers):
                #arcpy.AddMessage("deleting temporary boundary sliver layers")
                #arcpy.AddMessage(fc_slivers)
                arcpy.management.Delete(fc_slivers)
        
                
        return

class incorrectWetlandsCodes_poly(object):
    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "Incorrect Wetlands Codes - Polygonal"
        self.description = "Identifies wetland polygons with incorrect wetland codes, or null or blank values in the 'attribute' field. A bad attribute summary table is created and stored with your wetlands file geodatabase. Changes character 1 of QAQC_Code = 'C' if wetland code is bad. "
         

        self.canRunInBackground = False
        self.category = "Individual Polygonal Tools"

    
    def getParameterInfo(self):
        """Define parameter definitions"""
        # params [0]
        input_gdb = arcpy.Parameter(name = "input_gdb",
                                   displayName = "Input Geodatabase",
                                   datatype = "DEWorkspace",
                                   parameterType = "Required", #Required|Optional|Derived
                                   direction = "Input", #Input|Output
                                   )

        work_area = arcpy.Parameter(name = "work_area",
                                     displayName = "Work Area",
                                     datatype = "String",
                                     parameterType = "Required", #Required|Optional|Derived
                                     direction = "Input", #Input|Output
                                     )
        work_area.value = "CONUS"
        work_area.filter.type = "ValueList"
        work_area.filter.list = ["CONUS","AK","HI","PRVI","PacTrust"]

        #params = None
        
        return [input_gdb, work_area]

    def isLicensed(self):
        """Set whether tool is licensed to execute."""
        return True

    def updateParameters(self, parameters):
        """Modify the values and properties of parameters before internal
        validation is performed.  This method is called whenever a parameter
        has been changed."""
        
        return

    def updateMessages(self, parameters):
        """Modify the messages created by internal validation for each tool
        parameter.  This method is called after internal validation."""
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        wet_projects = os.path.join(input_gdb,"{work_area}_projects".format(work_area = work_area),"{work_area}_wet_projects".format(work_area=work_area))
        if not arcpy.Exists(wet_poly):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_poly.".format(work_area=work_area))

        return

    def execute(self, parameters, messages):
        """The source code of the tool."""

        arcpy.AddMessage("Incorrect Codes Version 1.4.3")
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        
        arcpy.management.Compact(input_gdb)

        start_count = arcpy.management.GetCount(wet_poly)
        arcpy.AddMessage("Begin Incorrect Codes tool")
        incorrectWetlandsCodes_poly.incorrectCodes(input_gdb,work_area)
        end_count = arcpy.management.GetCount(wet_poly)
        if start_count[0] != end_count[0]:
            arcpy.AddWarning("The number of features in {work_area}_wet_poly changed during the course of this tool.".format(work_area=work_area))

        
                        
        return

    @staticmethod
    def incorrectCodes(arg1,arg2):
        input_gdb = str(arg1)
        work_area = str(arg2)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))

        arcpy.management.MakeFeatureLayer(wet_poly,"wet_poly_lyr")

        fields = ['QAQC_CODE','ATTRIBUTE']

        edit = arcpy.da.Editor(input_gdb)
        edit.startEditing()
        edit.startOperation()

        with arcpy.da.UpdateCursor("wet_poly_lyr",fields) as cursor:
            #arcpy.AddMessage("update cursor processing...")
            for row in cursor:

                if row[0] is None:
                    row[0] = "NNNNNN"
                
                row[0] = "N"+row[0][1:]
                newCode = "N"+row[0][1:]

                if row[1] is not None:
                    row[1] = row[1].strip(" ")
                    
                cursor.updateRow(row)
                #arcpy.AddMessage("updated row")
                
        edit.stopOperation()
        edit.stopEditing(True)

        goodCodeList = []
        badCodeList = []

        #list_fields = ['ATTRIBUTE','QAQC_CODE']
        list_fields = ['ATTRIBUTE']
        #arcpy.AddMessage("begin ATTRIBUTE cursor")

        #arcpy.AddMessage(badCodeList)
        #arcpy.AddMessage(len(badCodeList))

        unique_codes = r"memory\UniqueCodes"

        arcpy.analysis.Frequency(wet_poly, unique_codes, "ATTRIBUTE", None)

        with arcpy.da.SearchCursor(unique_codes,list_fields) as cursor:
            for row in cursor:
                code = str(row[0])
                incorrectWetlandsCodes_poly.codeCheck(code,goodCodeList,badCodeList)


        if len(badCodeList) > 0:
            arcpy.management.MakeFeatureLayer(wet_poly,"wet_poly_lyr")
            superBadCodeList = []
            for code in badCodeList:
                if "'" in code:
                    arcpy.AddMessage("': {code}".format(code=code))
                    superBadCodeList.append(code)
                    badCodeList.remove(code)
                '''
                if r"\\" in code:
                    arcpy.AddMessage("\\: {code}".format(code=code))
                    superBadCodeList.append(code)
                    badCodeList.remove(code)
                '''

            if len(badCodeList) > 0:
                if len(badCodeList) == 1:
                    qry = """ATTRIBUTE IN {0})""".format(str(tuple(badCodeList))[:-2])
                else:
                    qry = """ATTRIBUTE IN {0}""".format(str(tuple(badCodeList)))
                #arcpy.AddMessage(qry)
                #arcpy.AddMessage("bad codes: {x1}".format(x1=badCodeList))
                #arcpy.AddMessage("super bad codes: {x1}".format(x1=superBadCodeList))
            '''
            if len(superBadCodeList) == 1:
                qry2 = """ATTRIBUTE IN {0})""".format(str(tuple(superBadCodeList))[:-2])
            arcpy.AddMessage(qry2)
            '''
        
            arcpy.management.SelectLayerByAttribute("wet_poly_lyr","NEW_SELECTION",qry)
            arcpy.management.SelectLayerByAttribute("wet_poly_lyr","ADD_TO_SELECTION","ATTRIBUTE IS NULL")
            arcpy.management.SelectLayerByAttribute("wet_poly_lyr","ADD_TO_SELECTION","ATTRIBUTE LIKE '%''%'")

            fields = ['QAQC_CODE', 'ATTRIBUTE']
            edit = arcpy.da.Editor(input_gdb)
            edit.startEditing()
            edit.startOperation()
            with arcpy.da.UpdateCursor("wet_poly_lyr",fields) as cursor:
                #arcpy.AddMessage("update cursor processing...")
                for row in cursor:
                    #if row[1] is None:
                    #    row[0] = "C"+row[0][1:]

                    if row[0] is None:
                        row[0] = "NNNNNN"

                    row[0] = "C"+row[0][1:]

                    cursor.updateRow(row)
                    #arcpy.AddMessage("updated row")

            edit.stopOperation()
            edit.stopEditing(True)
            
            arcpy.AddMessage("Incorrect Codes: {bCL}".format(bCL=badCodeList))

        
        

        return

    @staticmethod
    def codeCheck(code,goodCodeList,badCodeList):

        '''
        Early Checks
        Check if the code has already been determined to be good, cut down on processing time
        Check if code is long enough
        Check if code has invalid characters
        '''
        split_check = False

        #arcpy.AddMessage(code)

        if code in goodCodeList:
            #print("good, already checked this one")
            return 'good'
        if code in badCodeList:
            #print("no good, you already checked this one")
            badCodeList.append(code)
            return code

        if code is None:
            badCodeList.append(code)
            return code

        if code == 'None':
            badCodeList.append(code)
            return code

        if code == 'Pf':
            goodCodeList.append(code)
            return 'good'

        if len(code)<2:
            badCodeList.append(code)
            return code

        if code[:2] == 'Rp':
            badCodeList.append(code)
            return code

        if code[0] == 'P' and len(code)<4:
            #print("no good - not enough characters")
            badCodeList.append(code)
            return code
        elif code[0] == 'P' and code[3].isnumeric() and len(code)<5:
            #print("no good - not enough characters")
            badCodeList.append(code)
            return code
        elif code[0] in ('M','E','L','R') and len(code)<5:
            #print("no good - not enough characters")
            badCodeList.append(code)
            return code
        elif code[0] in ('M','E','L','R') and code[4].isnumeric() and len(code)<6:
            #print("no good - not enough characters")
            badCodeList.append(code)
            return code
        elif code[0] in ('M','E','L','R') and code[4].isalpha() and len(code)<5:
            #print("no good - not enough characters")
            badCodeList.append(code)
            return code

        if r'/' in code:
            split_check = True

            split_position = code.find(r'/')
            #arcpy.AddMessage("split!")
            #arcpy.AddMessage(code)
            #print(split_position)
            if code.count(r'/') > 1:
                #print("hey now, too many splits")
                
                badCodeList.append(code)
                return code
            if len(code)<split_position+3:
                #print("need more characters after the split")
                
                badCodeList.append(code)
                return code
            if code[split_position-1] == code[split_position+1]:
                if code[split_position-1].isnumeric():
                    badCodeList.append(code)
                    return code
            if code[split_position-3:split_position] == code[split_position+1:split_position+4]:
                badCodeList.append(code)
                return code
                
        if not code.isalnum() and r'/' not in code:
            #print("no good - invalid character(s)")
            badCodeList.append(code)
            return code

        #pd = codeParse(code)
        '''
        System/Subsystem Checks
        '''

        systemList = ['M','E','R','L','P']
        subSystemList = ['M1','M2','E1','E2','R1','R2','R3','R4','L1','L2']

        #system = pd['system']
        #print(system)

        if code[0] != 'P' and code[:2] not in subSystemList:
            #print("no good - system or subsystem")
            badCodeList.append(code)
            return code

        '''
        Class Checks
        '''
        
        classDict = {
            'M1': ['RB','UB','AB','RF'],
            'M2': ['AB','RF','RS','US'],
            'E1': ['RB','UB','AB','RF'],
            'E2': ['AB','RF','SB','RS','US','EM','SS','FO'],
            'R1': ['RB','UB','AB','SB','RS','US','EM'],
            'R2': ['UB','AB','RS','US','EM'],
            'R3': ['RB','UB','AB','RS','US'],
            'R4': ['SB'],
            'L1': ['RB','UB','AB'],
            'L2': ['RB','UB','AB','RS','US','EM'],
            'P': ['RB','UB','AB','US','ML','EM','SS','FO']
            }

        if code[0] == 'P':
            if code[1:3] not in classDict[code[:1]]:
                #print("no good - class")
                badCodeList.append(code)
                return code
        elif code[0] in ('M','E','L','R'):
            
            if code[2:4] not in classDict[code[:2]]:
                #print("no good - class")
                badCodeList.append(code)
                return code
        
        '''
        Subclass Checks
        '''
        
        subclassDict = {
            'RB': ['1','2'],
            'UB': ['1','2','3','4'],
            'AB': ['1','2','3','4'],
            'RF': ['1','2','3'],
            'SB': ['1','2','3','4','5','6','7'],
            'RS': ['1','2'],
            'US': ['1','2','3','4','5'],
            'ML': ['1','2'],
            'EM': ['1','2','5'],
            'SS': ['1','2','3','4','5','6','7'],
            'FO': ['1','2','3','4','5','6','7']
            }

        subclassExclusions = {
            'M1UB': ['4'],
            'M1AB': ['2','4'],
            'M1RF': ['2'],
            'M2AB': ['2','4'],
            'M2RF': ['2'],
            'M2US': ['5'],
            'E1AB': ['2'],
            'E1RF': ['2'],
            'E2AB': ['2'],
            'E2RF': ['1'],
            'E2SB': ['7'],
            'E2US': ['5'],
            'R1EM': ['1','5'],
            'R1SB': ['7'],
            'R2EM': ['1','5'],
            'R3UB': ['4'],
            'R3EM': ['1','5'],
            'R4EM': ['1','5'],
            'L2EM': ['1','5']
            }
        
        if code[0] in ('M','E','L','R'):
            
            if code[4].isnumeric() and code[4] not in subclassDict[code[2:4]]:
                #print("no good - subclass")
                badCodeList.append(code)
                return code
                
            if code[4].isnumeric() and code[:4] in subclassExclusions:
                if code[4] in subclassExclusions[code[:4]]:
                    #print("no good -subclass excluded")
                    badCodeList.append(code)
                    return code

            if code[4].isalpha() and code[2:4] in ('EM','SS','FO'):
                #print("no good - PEM, PSS, PFO require subclass")
                badCodeList.append(code)
                return code

            if code[4] == r'/' and code[2:4] in ('EM','SS','FO'):
                #print("no good - PEM, PSS, PFO require subclass")
                badCodeList.append(code)
                return code

        elif code[0] == 'P':

            if code[3].isnumeric() and code[3] not in subclassDict[code[1:3]]:
                #print("no good - subclass")
                badCodeList.append(code)
                return code

            if code[3].isalpha() and code[1:3] in ('EM','SS','FO'):
                #print("no good - PEM, PSS, PFO require subclass")
                badCodeList.append(code)
                return code

            if code[3] == r'/' and code[2:4] in ('EM','SS','FO'):
                #print("no good - PEM, PSS, PFO require subclass")
                badCodeList.append(code)
                return code

        '''
        Water Regime Checks
        '''
        
        waterRegimeDict = {
            'M1RB': ['L'],
            'M1UB': ['L'],
            'M1AB': ['L'],
            'M1RF': ['L'],
            'M2AB': ['M','N'],
            'M2RF': ['M','N'],
            'M2RS': ['M','N','P'],
            'M2US': ['M','N','P'],
            'E1RB': ['L'],
            'E1UB': ['L'],
            'E1AB': ['L'],
            'E1RF': ['L'],
            'E2AB': ['M','N'],
            'E2RF': ['M','N'],
            'E2SB': ['M','N','P'],
            'E2RS': ['M','N','P'],
            'E2US': ['M','N','P'],
            'E2EM': ['N','P'],
            'E2SS': ['M','N','P'],
            'E2FO': ['M','N','P'],
            'R1RB': ['T','V'],
            'R1UB': ['T','V'],
            'R1AB': ['Q','T','V'],
            'R1SB': ['Q'],
            'R1RS': ['Q'],
            'R1US': ['Q'],
            'R1EM': ['Q','T','V'],
            'R2UB': ['F','G','H'],
            'R2AB': ['C','F','G','H'],
            'R2RS': ['A','C'],
            'R2US': ['A','C','E','J'],
            'R2EM': ['F','G','H'],
            'R3RB': ['F','G','H'],
            'R3UB': ['F','G','H'],
            'R3AB': ['C','F','G','H'],
            'R3RS': ['A','C'],
            'R3US': ['A','C','E','J'],
            'R4SB': ['A','C','J'],
            'L1RB': ['V','G','H','K'],
            'L1UB': ['V','G','H','K'],
            'L1AB': ['V','G','H','K'],
            'L2RB': ['T','V','F','G','H','K'],
            'L2UB': ['T','V','F','G','H','K'],
            'L2AB': ['Q','C','T','V','F','G','H','K'],
            'L2RS': ['Q','A','C','J','K'],
            'L2US': ['Q','A','C','E','J','K'],
            'L2EM': ['Q','T','V','F','G','H','K'],
            'PRB': ['F','G','H','K'],
            'PUB': ['T','V','F','G','H','K'],
            'PAB': ['R','T','V','C','F','G','H','K'],
            'PUS': ['R','S','A','C','E','J','K'],
            'PML': ['B','C','D','E'],
            'PEM': ['R','S','T','V','A','B','C','D','E','F','G','H','J','K'],
            'PSS': ['R','S','T','V','A','B','C','D','E','F','G','H','J','K'],
            'PFO': ['R','S','T','V','A','B','C','D','E','F','G','H','K']
            }

        waterRegimeExclusions = {
            'M2AB3': ['N'],
            'E2AB3': ['N'],
            'E2EM5': ['N'],
            'E2SS1': ['M','N'],
            'E2SS2': ['M','N'],
            'E2SS3': ['M'],
            'E2SS4': ['M','N'],
            'E2SS6': ['M','N'],
            'E2SS7': ['M'],
            'E2FO1': ['M','N'],
            'E2FO2': ['M','N'],
            'E2FO3': ['M'],
            'E2FO4': ['M','N'],
            'E2FO6': ['M','N'],
            'E2FO7': ['M'],
            'R1AB1': ['Q'],
            'R1AB2': ['Q'],
            'R1US5': ['Q'],
            'R2AB1': ['C'],
            'R2AB2': ['C'],
            'R2US1': ['E'],
            'R2US2': ['E'],
            'R2US3': ['E'],
            'R2US4': ['A','C','J'],
            'R2US5': ['E'],
            'R3AB1': ['C'],
            'R3AB2': ['C'],
            'R3US1': ['E'],
            'R3US2': ['E'],
            'R3US3': ['E'],
            'R3US4': ['A','C','J'],
            'R3US5': ['E'],
            'R4SB6': ['A','J'],
            'L2AB1': ['C'],
            'L2AB2': ['C'],
            'L2US1': ['E'],
            'L2US2': ['E'],
            'L2US3': ['E'],
            'L2US4': ['A','C','J','K'],
            'L2US5': ['E'],
            'PAB1': ['R','C'],
            'PAB2': ['R','C'],
            'PUS1': ['E'],
            'PUS2': ['E'],
            'PUS3': ['E'],
            'PUS4': ['R','S','A','C','J','K'],
            'PUS5': ['R','S','E'],
            'PEM1': ['V','G','H'],
            'PEM2': ['R','S','A','B','C','D','E','J'],
            'PEM5': ['V','G','H'],
            'PSS1': ['V','G','H'],
            'PSS2': ['V','G','H'],
            'PSS3': ['T','V','F','G','H','J'],
            'PSS4': ['T','V','F','G','H','J'],
            'PSS5': ['R','S','A','B','C','D','E','J'],
            'PSS6': ['V','G','H'],
            'PSS7': ['T','V','F','G','H','J'],
            'PFO1': ['V','G','H'],
            'PFO2': ['V','G','H'],
            'PFO3': ['T','V','F','G','H'],
            'PFO4': ['T','V','F','G','H'],
            'PFO5': ['R','S','A','B','C','D','E'],
            'PFO6': ['V','G','H'],
            'PFO7': ['T','V','F','G','H']
            }
        if not split_check:

            if code[0] in ('M','E','L','R'):
                if code[4].isnumeric() and code[5] not in waterRegimeDict[code[:4]]:
                    #print("no good - water regime")
                    badCodeList.append(code)
                    return code
                elif code[4].isalpha() and code[4] not in waterRegimeDict[code[:4]]:
                    #print("no good - water regime")
                    badCodeList.append(code)
                    return code
                elif code[4].isnumeric() and code[:5] in waterRegimeExclusions:
                    if code[5] in waterRegimeExclusions[code[:5]]:
                        #print("no good -water regime exclusion")
                        badCodeList.append(code)
                        return code

            elif code[0] == 'P':
                if code[3].isnumeric() and code[4] not in waterRegimeDict[code[:3]]:
                    #print("no good - water regime")
                    badCodeList.append(code)
                    return code
                elif code[3].isalpha() and code[3] not in waterRegimeDict[code[:3]]:
                    #print("no good - water regime")
                    badCodeList.append(code)
                    return code
                elif code[3].isnumeric() and code[:4] in waterRegimeExclusions:
                    if code[4] in waterRegimeExclusions[code[:4]]:
                        #print("no good - water regime exclusion")
                        badCodeList.append(code)
                        return code

        mixedClassDict = {
            'AB': ['RF', 'RS', 'UB', 'US', 'FO', 'SS', 'EM'],
            'EM': ['AB', 'FO', 'ML', 'RS', 'SB', 'SS', 'UB', 'US'],
            'FO': ['AB', 'EM', 'ML', 'SS', 'UB', 'US'],
            'ML': ['EM', 'FO', 'SS'],
            'SS': ['AB', 'EM', 'FO', 'ML', 'UB', 'US'],
            'US': ['AB', 'EM', 'FO', 'RF', 'SS', 'RS'],
            'RF': ['AB', 'US'],
            'RS': ['AB', 'EM', 'US'],
            'UB': ['AB', 'EM', 'FO', 'SS']        
            }

        mixedSubclassDict = {
            '1': ['2', '3', '4', '5', '6', '7'],
            '2': ['1', '3', '4', '5', '6', '7'],
            '3': ['1', '2', '4', '5', '6', '7'],
            '4': ['1', '2', '3', '5', '6', '7'],
            '5': ['1', '2', '3', '4', '6', '7'],
            '6': ['1', '2', '3', '4', '5', '7'],
            '7': ['1', '2', '3', '4', '5', '6']
            }

        '''
        Special Modifier Checks
        '''
        
        modifierDict = {
            'sm': ['b','d','f','m','h','r','s','x'],
            'wcHS': ['1','2','3','4','5','6','0'],
            'wcpH': ['a','t','i'],
            's': ['g','n']
            }

        #illegal water regime and modifier combos#
        if 'Kh' in code:
            badCodeList.append(code)
            return code
        if 'Km' in code:
            badCodeList.append(code)
            return code

        if not split_check:
            
            if code[0] in ('M','E','L','R'):

                if code[4].isnumeric():
                    #print(len(code))
                    #print("water regime is: {x}".format(x=code[5]))
                    #print("modifiers are: {x}".format(x=code[6:]))
                    modifiers = code[6:]
                    if len(modifiers)==1:
                        if modifiers[0] in modifierDict['sm'] or modifiers[0] in modifierDict['wcHS'] or modifiers[0] in modifierDict['wcpH'] or modifiers in modifierDict['s']:
                            #print('good')
                            #arcpy.AddMessage('good')
                            pass
                        else:
                            #print('no good - special modifiers')
                            badCodeList.append(code)
                            return code
                    elif len(modifiers)==2:
                        if modifiers[0] in modifierDict['sm']:
                            if modifiers[1] in modifierDict['wcHS'] or modifiers[1] in modifierDict['wcpH'] or modifiers[1] in modifierDict['s']:
                                #print('good')
                                #arcpy.AddMessage('good')
                                pass
                            else:
                                #print('no good - special modifiers')
                                badCodeList.append(code)
                                return code
                        elif modifiers[0] in modifierDict['wcHS']:
                            if modifiers[1] in modifierDict['wcpH'] or modifiers[1] in modifierDict['s']:
                                #print('good')
                                #arcpy.AddMessage('good')
                                pass
                            else:
                                #print('no good - special modifiers')
                                badCodeList.append(code)
                                return code
                        elif modifiers[0] in modifierDict['wcpH']:
                            if modifiers[1] in modifierDict['s']:
                                #print('good')
                                #arcpy.AddMessage('good')
                                pass
                            else:
                                #print('no good - special modifiers')
                                badCodeList.append(code)
                                return code
                        else:
                            #print('no good - special modifiers')
                            badCodeList.append(code)
                            return code
                    elif len(modifiers)==3:
                        if modifiers[0] in modifierDict['sm']:
                            if modifiers[1] in modifierDict['wcHS']:
                                if modifiers[2] in modifierDict['wcpH'] or modifiers[2] in modifierDict['s']:
                                    #print('good')
                                    #arcpy.AddMessage('good')
                                    pass
                                else:
                                    #print('no good - special modifiers')
                                    badCodeList.append(code)
                                    return code
                            elif modifiers[1] in modifierDict['wcpH']:
                                if modifiers[2] in modifierDict['s']:
                                    #print('good')
                                    #arcpy.AddMessage('good')
                                    pass
                                else:
                                    #print('no good - special modifiers')
                                    badCodeList.append(code)
                                    return code
                        elif modifiers[0] in modifierDict['wcHS'] and modifiers[1] in modifierDict['wcpH'] and modifiers[2] in modifierDict['s']:
                            #print('good')
                            #arcpy.AddMessage('good')
                            pass
                        else:
                            #print('no good - special modifiers')
                            badCodeList.append(code)
                            return code
                                
                    elif len(modifiers)==4:
                        if modifiers[0] in modifierDict['sm'] and modifiers[1] in modifierDict['wcHS'] and modifiers[2] in modifierDict['wcpH'] and modifiers[3] in modifierDict['s']:
                            #print('good')
                            #arcpy.AddMessage('good')
                            pass
                        else:
                            #print('no good - special modifiers')
                            badCodeList.append(code)
                            return code
                    elif len(modifiers)>4:
                        #print('no good - special modifiers')
                        badCodeList.append(code)
                        return code

                else:
                    #print(len(code))
                    #print("water regime is: {x}".format(x=code[4]))
                    #print("modifiers are: {x}".format(x=code[5:]))
                    modifiers = code[5:]
                    
                    if len(modifiers)==1:
                        if modifiers[0] in modifierDict['sm'] or modifiers[0] in modifierDict['wcHS'] or modifiers[0] in modifierDict['wcpH'] or modifiers in modifierDict['s']:
                            #print('good')
                            #arcpy.AddMessage('good')
                            pass
                        else:
                            #print('no good - special modifiers')
                            badCodeList.append(code)
                            return code
                    elif len(modifiers)==2:
                        if modifiers[0] in modifierDict['sm']:
                            if modifiers[1] in modifierDict['wcHS'] or modifiers[1] in modifierDict['wcpH'] or modifiers[1] in modifierDict['s']:
                                #print('good')
                                #arcpy.AddMessage('good')
                                pass
                            else:
                                #print('no good - special modifiers')
                                badCodeList.append(code)
                                return code
                        elif modifiers[0] in modifierDict['wcHS']:
                            if modifiers[1] in modifierDict['wcpH'] or modifiers[1] in modifierDict['s']:
                                #print('good')
                                #arcpy.AddMessage('good')
                                pass
                            else:
                                #print('no good - special modifiers')
                                badCodeList.append(code)
                                return code
                        elif modifiers[0] in modifierDict['wcpH']:
                            if modifiers[1] in modifierDict['s']:
                                #print('good')
                                #arcpy.AddMessage('good')
                                pass
                            else:
                                #print('no good - special modifiers')
                                badCodeList.append(code)
                                return code
                        else:
                            #print('no good - special modifiers')
                            badCodeList.append(code)
                            return code
                    elif len(modifiers)==3:
                        if modifiers[0] in modifierDict['sm']:
                            if modifiers[1] in modifierDict['wcHS']:
                                if modifiers[2] in modifierDict['wcpH'] or modifiers[2] in modifierDict['s']:
                                    #print('good')
                                    #arcpy.AddMessage('good')
                                    pass
                                else:
                                    #print('no good - special modifiers')
                                    badCodeList.append(code)
                                    return code
                            elif modifiers[1] in modifierDict['wcpH']:
                                if modifiers[2] in modifierDict['s']:
                                    #print('good')
                                    #arcpy.AddMessage('good')
                                    pass
                                else:
                                    #print('no good - special modifiers')
                                    badCodeList.append(code)
                                    return code
                        elif modifiers[0] in modifierDict['wcHS'] and modifiers[1] in modifierDict['wcpH'] and modifiers[2] in modifierDict['s']:
                            #print('good')
                            #arcpy.AddMessage('good')
                            pass
                        else:
                            #print('no good - special modifiers')
                            badCodeList.append(code)
                            return code
                                
                    elif len(modifiers)==4:
                        if modifiers[0] in modifierDict['sm'] and modifiers[1] in modifierDict['wcHS'] and modifiers[2] in modifierDict['wcpH'] and modifiers[3] in modifierDict['s']:
                            #print('good')
                            #arcpy.AddMessage('good')
                            pass
                        else:
                            #print('no good - special modifiers')
                            badCodeList.append(code)
                            return code
                    elif len(modifiers)>4:
                        #print('no good - special modifiers')
                        badCodeList.append(code)
                        return code

                #print('good, i guess')
                return 'good'

            if code[0] == 'P':

                if code[3].isnumeric():
                    #print(len(code))
                    #print("water regime is: {x}".format(x=code[5]))
                    #print("modifiers are: {x}".format(x=code[6:]))
                    modifiers = code[5:]
                    if len(modifiers)==1:
                        if modifiers[0] in modifierDict['sm'] or modifiers[0] in modifierDict['wcHS'] or modifiers[0] in modifierDict['wcpH'] or modifiers in modifierDict['s']:
                            #print('good')
                            #arcpy.AddMessage('good')
                            pass
                        else:
                            #print('no good - special modifiers')
                            badCodeList.append(code)
                            return code
                    elif len(modifiers)==2:
                        if modifiers[0] in modifierDict['sm']:
                            if modifiers[1] in modifierDict['wcHS'] or modifiers[1] in modifierDict['wcpH'] or modifiers[1] in modifierDict['s']:
                                #print('good')
                                #arcpy.AddMessage('good')
                                pass
                            else:
                                #print('no good - special modifiers')
                                badCodeList.append(code)
                                return code
                        elif modifiers[0] in modifierDict['wcHS']:
                            if modifiers[1] in modifierDict['wcpH'] or modifiers[1] in modifierDict['s']:
                                #print('good')
                                #arcpy.AddMessage('good')
                                pass
                            else:
                                #print('no good - special modifiers')
                                badCodeList.append(code)
                                return code
                        elif modifiers[0] in modifierDict['wcpH']:
                            if modifiers[1] in modifierDict['s']:
                                #print('good')
                                #arcpy.AddMessage('good')
                                pass
                            else:
                                #print('no good - special modifiers')
                                badCodeList.append(code)
                                return code
                        else:
                            #print('no good - special modifiers')
                            badCodeList.append(code)
                            return code
                    elif len(modifiers)==3:
                        if modifiers[0] in modifierDict['sm']:
                            if modifiers[1] in modifierDict['wcHS']:
                                if modifiers[2] in modifierDict['wcpH'] or modifiers[2] in modifierDict['s']:
                                    #print('good')
                                    #arcpy.AddMessage('good')
                                    pass
                                else:
                                    #print('no good - special modifiers')
                                    badCodeList.append(code)
                                    return code
                            elif modifiers[1] in modifierDict['wcpH']:
                                if modifiers[2] in modifierDict['s']:
                                    #print('good')
                                    #arcpy.AddMessage('good')
                                    pass
                                else:
                                    #print('no good - special modifiers')
                                    badCodeList.append(code)
                                    return code
                        elif modifiers[0] in modifierDict['wcHS'] and modifiers[1] in modifierDict['wcpH'] and modifiers[2] in modifierDict['s']:
                            #print('good')
                            #arcpy.AddMessage('good')
                            pass
                        else:
                            #print('no good - special modifiers')
                            badCodeList.append(code)
                            return code
                                
                    elif len(modifiers)==4:
                        if modifiers[0] in modifierDict['sm'] and modifiers[1] in modifierDict['wcHS'] and modifiers[2] in modifierDict['wcpH'] and modifiers[3] in modifierDict['s']:
                            #print('good')
                            #arcpy.AddMessage('good')
                            pass
                        else:
                            #print('no good - special modifiers')
                            badCodeList.append(code)
                            return code
                    elif len(modifiers)>4:
                        #print('no good - special modifiers')
                        badCodeList.append(code)
                        return code

                else:
                    #print(len(code))
                    #print("water regime is: {x}".format(x=code[4]))
                    #print("modifiers are: {x}".format(x=code[5:]))
                    modifiers = code[4:]
                    
                    if len(modifiers)==1:
                        if modifiers[0] in modifierDict['sm'] or modifiers[0] in modifierDict['wcHS'] or modifiers[0] in modifierDict['wcpH'] or modifiers in modifierDict['s']:
                            #print('good')
                            #arcpy.AddMessage('good')
                            pass
                        else:
                            #print('no good - special modifiers')
                            badCodeList.append(code)
                            return code
                    elif len(modifiers)==2:
                        if modifiers[0] in modifierDict['sm']:
                            if modifiers[1] in modifierDict['wcHS'] or modifiers[1] in modifierDict['wcpH'] or modifiers[1] in modifierDict['s']:
                                #print('good')
                                #arcpy.AddMessage('good')
                                pass
                            else:
                                #print('no good - special modifiers')
                                badCodeList.append(code)
                                return code
                        elif modifiers[0] in modifierDict['wcHS']:
                            if modifiers[1] in modifierDict['wcpH'] or modifiers[1] in modifierDict['s']:
                                #print('good')
                                #arcpy.AddMessage('good')
                                pass
                            else:
                                #print('no good - special modifiers')
                                badCodeList.append(code)
                                return code
                        elif modifiers[0] in modifierDict['wcpH']:
                            if modifiers[1] in modifierDict['s']:
                                #print('good')
                                #arcpy.AddMessage('good')
                                pass
                            else:
                                #print('no good - special modifiers')
                                badCodeList.append(code)
                                return code
                        else:
                            #print('no good - special modifiers')
                            badCodeList.append(code)
                            return code
                    elif len(modifiers)==3:
                        if modifiers[0] in modifierDict['sm']:
                            if modifiers[1] in modifierDict['wcHS']:
                                if modifiers[2] in modifierDict['wcpH'] or modifiers[2] in modifierDict['s']:
                                    #print('good')
                                    #arcpy.AddMessage('good')
                                    pass
                                else:
                                    #print('no good - special modifiers')
                                    badCodeList.append(code)
                                    return code
                            elif modifiers[1] in modifierDict['wcpH']:
                                if modifiers[2] in modifierDict['s']:
                                    #print('good')
                                    #arcpy.AddMessage('good')
                                    pass
                                else:
                                    #print('no good - special modifiers')
                                    badCodeList.append(code)
                                    return code
                        elif modifiers[0] in modifierDict['wcHS'] and modifiers[1] in modifierDict['wcpH'] and modifiers[2] in modifierDict['s']:
                            #print('good')
                            #arcpy.AddMessage('good')
                            pass
                        else:
                            #print('no good - special modifiers')
                            badCodeList.append(code)
                            return code
                                
                    elif len(modifiers)==4:
                        if modifiers[0] in modifierDict['sm'] and modifiers[1] in modifierDict['wcHS'] and modifiers[2] in modifierDict['wcpH'] and modifiers[3] in modifierDict['s']:
                            #print('good')
                            #arcpy.AddMessage('good')
                            pass
                        else:
                            #print('no good - special modifiers')
                            badCodeList.append(code)
                            return code
                    elif len(modifiers)>4:
                        #print('no good - special modifiers')
                        badCodeList.append(code)
                        return code
            

                goodCodeList.append(code)

                #print('good, i guess')
                return 'good'

        #This subsequent Else statement begins the process of checking split codes
        else:
            
            split_position = code.find(r'/')
            #arcpy.AddMessage(code)
            #arcpy.AddMessage(len(code))
            #arcpy.AddMessage(split_position)
            
            system = code[0]
            if system == 'P':
                subsystem = ''
            else:
                subsystem = code[1]

            if code[split_position-1].isnumeric():
                class1 = code[split_position-3:split_position-1]
                subclass1 = code[split_position-1]
            else:
                class1 = code[split_position-2:split_position]
                subclass1 = ''

            if code[split_position+1].isnumeric():
                class2 = ''
                subclass2 = code[split_position+1]
                waterregime = code[split_position+2]
                if len(code) >= split_position+3:
                    modifiers = code[split_position+3:]
            else:
                #arcpy.AddMessage(len(code))
                #arcpy.AddMessage(split_position)
                if len(code) < split_position+4:
                    badCodeList.append(code)
                    #arcpy.AddMessage('code: {x}'.format(x=code))
                    return code
                else:
                    class2 = code[split_position+1:split_position+3]
                    
                if code[split_position+3].isnumeric():
                    if len(code) < split_position+5:
                        badCodeList.append(code)
                        return code
                    else:                
                        subclass2 = code[split_position+3]
                        waterregime = code[split_position+4]
                        if len(code) >= split_position+5:
                            modifiers = code[split_position+5:]
                else:
                    subclass2 = ''
                    waterregime = code[split_position+3]
                    if len(code) >= split_position+4:
                        modifiers = code[split_position+4:]

            #arcpy.AddMessage("class1: {x1},\nsubclass1: {x2},\nclass2: {x3},\nsubclass2: {x4},\nwater regime: {x5},\nspecial modifiers:\n{x6}".format(x1=class1, x2=subclass1, x3=class2, x4=subclass2, x5=waterregime, x6=modifiers))

            sysSubsystem = system+subsystem

            baseCode1 = system+subsystem+class1
            if class2 == '':
                baseCode2 = system+subsystem+class1
            else:
                baseCode2 = system+subsystem+class2

            subCode1 = system+subsystem+class1+subclass1

            if class2 == '': 
                subCode2 = system+subsystem+class1+subclass2
            else: 
                subCode2 = system+subsystem+class2+subclass2

            #arcpy.AddMessage("basecode1 "+baseCode1)
            #arcpy.AddMessage("basecode2 "+baseCode2)

            if class1 not in classDict[sysSubsystem]:
                #print("no good - class")
                badCodeList.append(code)
                return code
            if class2 != '' and class2 not in classDict[sysSubsystem]:
                #print("no good - second class")
                badCodeList.append(code)
                return code

            #check for class agreements here
            if class2 != '' and class1 in mixedClassDict and class2 not in mixedClassDict[class1]:
                #print("no good - mixed classes not allowed")
                badCodeList.append(code)
                return code

            #subclass checks

            if subclass1 != '' and class1 in subclassDict:
                if subclass1 not in subclassDict[class1]:
                    #print("no good - subclass")
                    badCodeList.append(code)
                    return code
                
            if subclass1.isnumeric() and baseCode1 in subclassExclusions:
                if subclass1 in subclassExclusions[baseCode1]:
                    #print("no good -subclass excluded")
                    badCodeList.append(code)
                    return code
                
            if subclass1 == '' and class1 in ('EM','SS','FO'):
                #print("no good - EM, SS, FO require subclass")
                badCodeList.append(code)
                return code

            if subclass1.isalpha() and class1 in ('EM','SS','FO'):
                #print("no good - EM, SS, FO require subclass")
                badCodeList.append(code)
                return code

            if class2 != '':
                if subclass2 != '' and class2 in subclassDict:
                    if subclass2 not in subclassDict[class2]:
                        #print("no good - second subclass")
                        badCodeList.append(code)
                        return code

                if subclass2.isnumeric() and baseCode2 in subclassExclusions:
                    if subclass2 in subclassExclusions[baseCode2]:
                        #print("no good -subclass excluded")
                        badCodeList.append(code)
                        return code

                if subclass2 == '' and class2 in ('EM','SS','FO'):
                    #print("no good - EM, SS, FO require subclass")
                    badCodeList.append(code)
                    return code

            if class2 == '':
                if subclass2 != '' and class1 in subclassDict:
                    if subclass2 not in subclassDict[class1]:
                        #print("no good - second subclass")
                        badCodeList.append(code)
                        return code

                if subclass2.isnumeric() and baseCode2 in subclassExclusions:
                    if subclass2 in subclassExclusions[baseCode1]:
                        #print("no good -subclass excluded")
                        badCodeList.append(code)
                        return code

                if subclass2.isalpha() and class1 in ('EM','SS','FO'):
                    #print("no good - EM, SS, FO require subclass")
                    badCodeList.append(code)
                    return code

            #check for subclass agreements here

            if waterregime not in waterRegimeDict[baseCode1] or waterregime not in waterRegimeDict[baseCode2]:
                #print("no good - water regime")
                badCodeList.append(code)
                return code

            if subCode1 in waterRegimeExclusions:
                if waterregime in waterRegimeExclusions[subCode1]:
                    #print("no good -water regime exclusion")
                    badCodeList.append(code)
                    return code

            if subCode2 in waterRegimeExclusions:
                if waterregime in waterRegimeExclusions[subCode2]:
                    #print("no good -water regime exclusion")
                    badCodeList.append(code)
                    return code

            #special modifier checks

            if len(modifiers)==1:
                if modifiers[0] in modifierDict['sm'] or modifiers[0] in modifierDict['wcHS'] or modifiers[0] in modifierDict['wcpH'] or modifiers in modifierDict['s']:
                    #print('good')
                    #arcpy.AddMessage('good')
                    pass
                else:
                    #print('no good - special modifiers')
                    badCodeList.append(code)
                    return code
            elif len(modifiers)==2:
                if modifiers[0] in modifierDict['sm']:
                    if modifiers[1] in modifierDict['wcHS'] or modifiers[1] in modifierDict['wcpH'] or modifiers[1] in modifierDict['s']:
                        #print('good')
                        #arcpy.AddMessage('good')
                        pass
                    else:
                        #print('no good - special modifiers')
                        badCodeList.append(code)
                        return code
                elif modifiers[0] in modifierDict['wcHS']:
                    if modifiers[1] in modifierDict['wcpH'] or modifiers[1] in modifierDict['s']:
                        #print('good')
                        #arcpy.AddMessage('good')
                        pass
                    else:
                        #print('no good - special modifiers')
                        badCodeList.append(code)
                        return code
                elif modifiers[0] in modifierDict['wcpH']:
                    if modifiers[1] in modifierDict['s']:
                        #print('good')
                        #arcpy.AddMessage('good')
                        pass
                    else:
                        #print('no good - special modifiers')
                        badCodeList.append(code)
                        return code
                else:
                    #print('no good - special modifiers')
                    badCodeList.append(code)
                    return code
            elif len(modifiers)==3:
                if modifiers[0] in modifierDict['sm']:
                    if modifiers[1] in modifierDict['wcHS']:
                        if modifiers[2] in modifierDict['wcpH'] or modifiers[2] in modifierDict['s']:
                            #print('good')
                            #arcpy.AddMessage('good')
                            pass
                        else:
                            #print('no good - special modifiers')
                            badCodeList.append(code)
                            return code
                    elif modifiers[1] in modifierDict['wcpH']:
                        if modifiers[2] in modifierDict['s']:
                            #print('good')
                            #arcpy.AddMessage('good')
                            pass
                        else:
                            #print('no good - special modifiers')
                            badCodeList.append(code)
                            return code
                elif modifiers[0] in modifierDict['wcHS'] and modifiers[1] in modifierDict['wcpH'] and modifiers[2] in modifierDict['s']:
                    #print('good')
                    #arcpy.AddMessage('good')
                    pass
                else:
                    #print('no good - special modifiers')
                    badCodeList.append(code)
                    return code
                        
            elif len(modifiers)==4:
                if modifiers[0] in modifierDict['sm'] and modifiers[1] in modifierDict['wcHS'] and modifiers[2] in modifierDict['wcpH'] and modifiers[3] in modifierDict['s']:
                    #print('good')
                    #arcpy.AddMessage('good')
                    pass
                else:
                    #print('no good - special modifiers')
                    badCodeList.append(code)
                    return code
            elif len(modifiers)>4:
                #print('no good - special modifiers')
                badCodeList.append(code)
                return code

            

            goodCodeList.append(code)

            #print('good, i guess')
            return 'good'

        
        

class combinedPolyTools(object):
    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "Combined Wetland Polygonal Tools"
        self.description = "This tool runs all the polygonal QAQC Tools."
        self.canRunInBackground = False
        self.category = "Combined Tools"
        #self.QAQCCODERESET = QAQCcodeReset()
        #self.SLIVERWETLANDS = sliverWetlands()

    
    def getParameterInfo(self):
        """Define parameter definitions"""
        # params [0]
        input_gdb = arcpy.Parameter(name = "input_gdb",
                                   displayName = "Input Geodatabase",
                                   datatype = "DEWorkspace",
                                   parameterType = "Required", #Required|Optional|Derived
                                   direction = "Input", #Input|Output
                                   )

        work_area = arcpy.Parameter(name = "work_area",
                                     displayName = "Work Area",
                                     datatype = "String",
                                     parameterType = "Required", #Required|Optional|Derived
                                     direction = "Input", #Input|Output
                                     )

        work_area.value = "CONUS"
        work_area.filter.type = "ValueList"
        work_area.filter.list = ["CONUS","AK","HI","PRVI","PacTrust"]

        verifier_name = arcpy.Parameter(name = "verifier_name",
                                        displayName = "Verified By:",
                                        datatype = "String",
                                        parameterType = "Required",
                                        direction = "Input",
                                        )

        save_summary_table = arcpy.Parameter(name = "save_summary_table",
                                             displayName = "Save Summary Table?",
                                             datatype = "GPBoolean",
                                             parameterType = "Required",
                                             direction = "Input",
                                             )
        save_summary_table.value = True
        
        intermediate_data = arcpy.Parameter(name = "intermediate_data_type",
                                            displayName = "Temporary Data Type",
                                            datatype = "GPBoolean",
                                            parameterType = "Required",
                                            direction = "Input"
                                            )

        intermediate_data.value = False
        
        project_division = arcpy.Parameter(name = "project_division_number",
                                           displayName = "Number of Processing Divisions",
                                           datatype = "GPLong",
                                           parameterType = "Required",
                                           direction = "Input")

        project_division.value = 1
        project_division.filter.type = "ValueList"
        project_division.filter.list = [1,4,9,16,25]

        #params = None
        return [input_gdb,work_area,verifier_name,save_summary_table,intermediate_data,project_division]

    def isLicensed(self):
        """Set whether tool is licensed to execute."""
        return True

    def updateParameters(self, parameters):
        """Modify the values and properties of parameters before internal
        validation is performed.  This method is called whenever a parameter
        has been changed."""

        return

    def updateMessages(self, parameters):
        """Modify the messages created by internal validation for each tool
        parameter.  This method is called after internal validation."""
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        wet_projects = os.path.join(input_gdb,"{work_area}_projects".format(work_area = work_area),"{work_area}_wet_projects".format(work_area=work_area))
        if not arcpy.Exists(wet_poly):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_poly.".format(work_area=work_area))
        if not arcpy.Exists(wet_projects):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_projects.".format(work_area=work_area))

        return

    def execute(self, parameters, messages):
        """The source code of the tool."""

        arcpy.AddMessage("Combined Wetland Polygonal Tools Version 1.4.3")
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        wet_projects = os.path.join(input_gdb,"{work_area}_projects".format(work_area = work_area),"{work_area}_wet_projects".format(work_area=work_area))

        arcpy.management.Compact(input_gdb)

        verifier_name = str(parameters[2].value)
        save_summary_table = str(parameters[3].value)
        intermediate_data = parameters[4].value
        project_divisions = parameters[5].value

        #arcpy.AddMessage(verifier_name)
        #arcpy.AddMessage(save_summary_table)

        arcpy.AddMessage("Begin QAQC Code Reset")
        QAQCcodeReset_poly.QAQCcodeReset_tool(input_gdb,work_area)
        arcpy.AddMessage("Begin Wetland Type Calculation")
        wetlandTypeCalc_poly.wetlandTypeCalc(input_gdb,work_area)
        arcpy.AddMessage("Begin Create NWI_ID Tool")
        create_NWIID.create_NWIID_tool(input_gdb,work_area,"wet")
        arcpy.AddMessage("Begin Sliver Wetlands")
        sliverWetlands_poly.sliverWetlands(input_gdb,work_area)
        arcpy.AddMessage("Begin Overlapping Wetlands")
        overlappingWetlands_poly.overlappingWetlands(input_gdb,work_area)
        arcpy.AddMessage("Begin Lake and Pond Size")
        lakeAndPondSize.lakeAndPondSize(input_gdb,work_area,intermediate_data)
        arcpy.AddMessage("Begin Adjacent Wetlands")
        adjacentWetlands.adjacentWetlands(input_gdb,work_area,intermediate_data)
        arcpy.AddMessage("Begin Sliver Uplands")
        sliverUplands_poly.sliverUplands(input_gdb,work_area,intermediate_data,project_divisions)
        arcpy.AddMessage("Begin Incorrect Codes")
        incorrectWetlandsCodes_poly.incorrectCodes(input_gdb,work_area)
        
        arcpy.AddMessage("Begin QAQC Summary")
        QAQCSummary_poly.QAQCSummary_tool(input_gdb,work_area,verifier_name,save_summary_table)
        return
        
        
        

class QAQCcodeReset_line(object):
    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "QAQC Code Reset - Linear"
        self.description = "This tool calculates the QAQC_Code to be 'NNNNN'. \n This erases all recorded errors in the dataset and properly attributes the field for use by all other models. "
        self.canRunInBackground = False
        self.category = "Individual Linear Tools"

    
    def getParameterInfo(self):
        """Define parameter definitions"""
        # params [0]
        input_gdb = arcpy.Parameter(name = "input_gdb",
                                   displayName = "Input Geodatabase",
                                   datatype = "DEWorkspace",
                                   parameterType = "Required", #Required|Optional|Derived
                                   direction = "Input", #Input|Output
                                   )

        work_area = arcpy.Parameter(name = "work_area",
                                     displayName = "Work Area",
                                     datatype = "String",
                                     parameterType = "Required", #Required|Optional|Derived
                                     direction = "Input", #Input|Output
                                     )
        work_area.value = "CONUS"
        work_area.filter.type = "ValueList"
        work_area.filter.list = ["CONUS","AK","HI","PRVI","PacTrust"]

        #params = None
        return [input_gdb,work_area]

    def isLicensed(self):
        """Set whether tool is licensed to execute."""
        return True

    def updateParameters(self, parameters):
        """Modify the values and properties of parameters before internal
        validation is performed.  This method is called whenever a parameter
        has been changed."""

        return

    def updateMessages(self, parameters):
        """Modify the messages created by internal validation for each tool
        parameter.  This method is called after internal validation."""
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_line = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_line".format(work_area=work_area))
        wet_projects = os.path.join(input_gdb,"{work_area}_projects".format(work_area = work_area),"{work_area}_wet_projects".format(work_area=work_area))
        if not arcpy.Exists(wet_line):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_line.".format(work_area=work_area))
        

        return

    def execute(self, parameters, messages):
        """The source code of the tool."""

        arcpy.AddMessage("QAQC Code Reset Linear Version 1.4.3")
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        
        arcpy.management.Compact(input_gdb)

        '''
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        qaqc_code = "'NNNNNN'"
        arcpy.management.CalculateField(wet_poly,"QAQC_CODE",qaqc_code,"PYTHON_9.3")
        '''
        arcpy.AddMessage("Begin linear QAQC Code Reset")
        QAQCcodeReset_line.QAQCcodeReset_tool(input_gdb,work_area)
        return

    @staticmethod
    def QAQCcodeReset_tool(arg1,arg2):
        input_gdb = str(arg1)
        work_area = str(arg2)
        wet_line = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_line".format(work_area=work_area))

        qaqc_code = "'NNNNN'"
        
        arcpy.management.CalculateField(wet_line,"QAQC_CODE",qaqc_code,"PYTHON_9.3")
        return


class wetlandTypeCalc_line(object):
    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "Wetland Type Calculation - Linear"
        self.description = "This tool calculates the 'wetland_type' field based on the wetland code in the 'attribute' field. The 'wetland_type' field provides a general description of the wetland type and is used in the cartographic representation of the different wetland types on the Wetlands Mapper."

        self.canRunInBackground = False
        self.category = "Individual Linear Tools"

    
    def getParameterInfo(self):
        """Define parameter definitions"""
        # params [0]
        input_gdb = arcpy.Parameter(name = "input_gdb",
                                   displayName = "Input Geodatabase",
                                   datatype = "DEWorkspace",
                                   parameterType = "Required", #Required|Optional|Derived
                                   direction = "Input", #Input|Output
                                   )

        work_area = arcpy.Parameter(name = "work_area",
                                     displayName = "Work Area",
                                     datatype = "String",
                                     parameterType = "Required", #Required|Optional|Derived
                                     direction = "Input", #Input|Output
                                     )
        work_area.value = "CONUS"
        work_area.filter.type = "ValueList"
        work_area.filter.list = ["CONUS","AK","HI","PRVI","PacTrust"]

        #params = None
        
        return [input_gdb, work_area]

    def isLicensed(self):
        """Set whether tool is licensed to execute."""
        return True

    def updateParameters(self, parameters):
        """Modify the values and properties of parameters before internal
        validation is performed.  This method is called whenever a parameter
        has been changed."""

        return

    def updateMessages(self, parameters):
        """Modify the messages created by internal validation for each tool
        parameter.  This method is called after internal validation."""
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        wet_line = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_line".format(work_area=work_area))
        wet_projects = os.path.join(input_gdb,"{work_area}_projects".format(work_area = work_area),"{work_area}_wet_projects".format(work_area=work_area))
        if not arcpy.Exists(wet_poly):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_poly.".format(work_area=work_area))
        if not arcpy.Exists(wet_line):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_line.".format(work_area=work_area))
        if not arcpy.Exists(wet_projects):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_projects.".format(work_area=work_area))

        return

    def execute(self, parameters, messages):
        """The source code of the tool."""

        arcpy.AddMessage("Wetland Type Linear Version 1.4.3")
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        #wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))

        arcpy.management.Compact(input_gdb)

        arcpy.AddMessage("Begin linear Wetland Type Calculation tool")
        wetlandTypeCalc_line.wetlandTypeCalc(input_gdb,work_area)
        
        return

    @staticmethod
    def wetlandTypeCalc(arg1,arg2):

        input_gdb = str(arg1)
        work_area = str(arg2)
        wet_line = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_line".format(work_area=work_area))
        #try calculate field option here
        '''def wetlandtype( code):
            if code[:3] == "PFO":
                return "Freshwater Forested/Shrub Wetland"
            elif code[:3] == "PSS":
                return "Freshwater Forested/Shrub Wetland"
            elif code[:3] == "PEM":
                return "Freshwater Emergent Wetland"
            elif code[:3] == "PUB":
                return "Freshwater Pond"
            elif code[:3] == "PAB":
                return "Freshwater Pond"
            elif code[:3] == "PUS":
                return "Freshwater Pond"
            elif code[:2] == "E2":
                return "Estuarine and Marine Wetland"
            elif code[:2] == "M2":
                return "Estuarine and Marine Wetland"
            elif code[:2] == "E1":
                return "Estuarine and Marine Deepwater"
            elif code[:2] == "M1":
                return "Estuarine and Marine Deepwater"
            elif code[:1] == "R":
                return "Riverine"
            elif code[:1] == "L":
                return "Lake"
            else:
                return "Other"

        arcpy.CalculateField_management(wet_poly, "WETLAND_TYPE", "wetlandtype( !ATTRIBUTE!)", "PYTHON")
        '''
        #try update row option here
        fields = ['ATTRIBUTE','WETLAND_TYPE']
        #arcpy.AddMessage("begin cursor")
        edit = arcpy.da.Editor(input_gdb)
        edit.startEditing()
        edit.startOperation()
        with arcpy.da.UpdateCursor(wet_line,fields) as cursor:
            #arcpy.AddMessage("update cursor processing...")
            for row in cursor:
                #arcpy.AddMessage("row[0]: {x1}, row[1]: {x2}".format(x1=row[0],x2=row[1]))
                if row[0] is None:
                    row[1] = "NA"
                elif len(row[0])<2:
                    row[1] = "NA"
                elif row[0] == "Pf":
                    row[1] = "Other"
                elif row[0][:2] == "Rp":
                    row[1] = "Riparian"
                elif row[0][:3] == "PFO":
                    row[1] = "Freshwater Forested/Shrub Wetland"
                elif row[0][:3] == "PSS":
                    row[1] = "Freshwater Forested/Shrub Wetland"
                elif row[0][:3] == "PEM":
                    row[1] = "Freshwater Emergent Wetland"
                elif row[0][:3] == "PUB":
                    row[1] = "Freshwater Pond"
                elif row[0][:3] == "PAB":
                    row[1] = "Freshwater Pond"
                elif row[0][:3] == "PUS":
                    row[1] = "Freshwater Pond"
                elif row[0][:2] == "E2":
                    row[1] = "Estuarine and Marine Wetland"
                elif row[0][:2] == "M2":
                    row[1] = "Estuarine and Marine Wetland"
                elif row[0][:2] == "E1":
                    row[1] = "Estuarine and Marine Deepwater"
                elif row[0][:2] == "M1":
                    row[1] = "Estuarine and Marine Deepwater"
                elif row[0][:1] == "R":
                    row[1] = "Riverine"
                elif row[0][:1] == "L":
                    row[1] = "Lake"
                else:
                    row[1] = "Other"
                cursor.updateRow(row)
                #arcpy.AddMessage("updated row")
        edit.stopOperation()
        edit.stopEditing(True)
                                        
        return

class sliverWetlands_line(object):
    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "Sliver Wetlands - Linear"
        self.description = "This tool identifies wetland lines shorter than 2.66 meters, and changes character 3 of QAQC_Code = 'S'. \n These wetland features fall below the minimum mapping standard for wetlands and should be reviewed. Actual wetland features flagged as sliver wetlands can be justified as correct in the comments field of the QAQC_table. These comments will only be saved if the ‘Append to History Table’ box is checked prior to running the All_QAQC_Checks model."
         

        self.canRunInBackground = False
        self.category = "Individual Linear Tools"

    
    def getParameterInfo(self):
        """Define parameter definitions"""
        # params [0]
        input_gdb = arcpy.Parameter(name = "input_gdb",
                                   displayName = "Input Geodatabase",
                                   datatype = "DEWorkspace",
                                   parameterType = "Required", #Required|Optional|Derived
                                   direction = "Input", #Input|Output
                                   )

        work_area = arcpy.Parameter(name = "work_area",
                                     displayName = "Work Area",
                                     datatype = "String",
                                     parameterType = "Required", #Required|Optional|Derived
                                     direction = "Input", #Input|Output
                                     )
        work_area.value = "CONUS"
        work_area.filter.type = "ValueList"
        work_area.filter.list = ["CONUS","AK","HI","PRVI","PacTrust"]

        #params = None
        
        return [input_gdb, work_area]
    
    def isLicensed(self):
        """Set whether tool is licensed to execute."""
        return True

    def updateParameters(self, parameters):
        """Modify the values and properties of parameters before internal
        validation is performed.  This method is called whenever a parameter
        has been changed."""

        return

    def updateMessages(self, parameters):
        """Modify the messages created by internal validation for each tool
        parameter.  This method is called after internal validation."""
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        wet_line = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_line".format(work_area=work_area))
        wet_projects = os.path.join(input_gdb,"{work_area}_projects".format(work_area = work_area),"{work_area}_wet_projects".format(work_area=work_area))
        if not arcpy.Exists(wet_line):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_line.".format(work_area=work_area))
        

        return

    def execute(self, parameters, messages):
        """The source code of the tool."""

        arcpy.AddMessage("Sliver Wetlands Linear Version 1.4.3")
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        #wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))

        arcpy.management.Compact(input_gdb)

        '''
        qaqc_code = "'NNNNNN'"
        
        arcpy.CalculateField_management(wet_poly,"ACRES","!shape.area@acres!","PYTHON","")
        
        #Try the cursor method here
        fields = ['QAQC_CODE','ACRES']
        arcpy.AddMessage("begin cursor")
        edit = arcpy.da.Editor(input_gdb)
        edit.startEditing()
        edit.startOperation()
        with arcpy.da.UpdateCursor(wet_poly,fields) as cursor:
            arcpy.AddMessage("update cursor processing...")
            for row in cursor:
                arcpy.AddMessage("row[0]: {x1}, row[1]: {x2}".format(x1=row[0],x2=row[1]))
                if row[1] < .01:
                    arcpy.AddMessage("row[1]<.01")
                    row[0] = row[0][:2]+"S"+row[0][3:]
                    newCode = row[0][:2]+"S"+row[0][3:]
                    arcpy.AddMessage("newCode = {x3}".format(x3=newCode))
                    cursor.updateRow(row)
                    arcpy.AddMessage("updated row")

        edit.stopOperation()
        edit.stopEditing(True)
        '''
        arcpy.AddMessage("Begin Linear Sliver Wetlands tool")
        sliverWetlands_line.sliverWetlands(input_gdb,work_area)
        return

    @staticmethod
    def sliverWetlands(arg1,arg2):
        input_gdb = str(arg1)
        work_area = str(arg2)
        wet_line = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_line".format(work_area=work_area))

        qaqc_code = "'NNNNN'"
        
        #arcpy.CalculateField_management(wet_poly,"ACRES","!shape.area@acres!","PYTHON","")
        
        fields = ['QAQC_CODE','Shape_Length']
        #arcpy.AddMessage("begin cursor")
        edit = arcpy.da.Editor(input_gdb)
        edit.startEditing()
        edit.startOperation()
        with arcpy.da.UpdateCursor(wet_line,fields) as cursor:
            #arcpy.AddMessage("update cursor processing...")
            for row in cursor:
                #arcpy.AddMessage("row[0]: {x1}, row[1]: {x2}".format(x1=row[0],x2=row[1]))
                if row[0] is None:
                    row[0] = "NNNNN"
                if row[1] < 2.6666:
                    #arcpy.AddMessage("row[1]<.01")
                    row[0] = row[0][:2]+"S"+row[0][3:]
                    newCode = row[0][:2]+"S"+row[0][3:]
                    #arcpy.AddMessage("newCode = {x3}".format(x3=newCode))
                    cursor.updateRow(row)
                    #arcpy.AddMessage("updated row")

        edit.stopOperation()
        edit.stopEditing(True)
    
        return


class incorrectWetlandsCodes_line(object):
    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "Incorrect Wetlands Codes - Linear"
        self.description = "Identifies wetland lines with incorrect wetland codes, or null or blank values in the 'attribute' field. A bad attribute summary table is created and stored with your wetlands file geodatabase. Changes character 1 of QAQC_Code = 'C' if wetland code is bad. "
         

        self.canRunInBackground = False
        self.category = "Individual Linear Tools"

    
    def getParameterInfo(self):
        """Define parameter definitions"""
        # params [0]
        input_gdb = arcpy.Parameter(name = "input_gdb",
                                   displayName = "Input Geodatabase",
                                   datatype = "DEWorkspace",
                                   parameterType = "Required", #Required|Optional|Derived
                                   direction = "Input", #Input|Output
                                   )

        work_area = arcpy.Parameter(name = "work_area",
                                     displayName = "Work Area",
                                     datatype = "String",
                                     parameterType = "Required", #Required|Optional|Derived
                                     direction = "Input", #Input|Output
                                     )
        work_area.value = "CONUS"
        work_area.filter.type = "ValueList"
        work_area.filter.list = ["CONUS","AK","HI","PRVI","PacTrust"]

        #params = None
        
        return [input_gdb, work_area]

    def isLicensed(self):
        """Set whether tool is licensed to execute."""
        return True

    def updateParameters(self, parameters):
        """Modify the values and properties of parameters before internal
        validation is performed.  This method is called whenever a parameter
        has been changed."""

        return

    def updateMessages(self, parameters):
        """Modify the messages created by internal validation for each tool
        parameter.  This method is called after internal validation."""
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        wet_line = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_line".format(work_area=work_area))
        wet_projects = os.path.join(input_gdb,"{work_area}_projects".format(work_area = work_area),"{work_area}_wet_projects".format(work_area=work_area))
        if not arcpy.Exists(wet_line):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_line.".format(work_area=work_area))


        return

    def execute(self, parameters, messages):
        """The source code of the tool."""

        arcpy.AddMessage("Incorrect Wetlands Codes Linear Version 1.4.3")
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        #wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        
        arcpy.management.Compact(input_gdb)
        
        arcpy.AddMessage("Begin linear Incorrect Wetlands Codes tool")
        incorrectWetlandsCodes_line.incorrectCodes(input_gdb,work_area)
                        
        return

    @staticmethod
    def incorrectCodes(arg1,arg2):
        input_gdb = str(arg1)
        work_area = str(arg2)
        wet_line = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_line".format(work_area=work_area))

        arcpy.management.MakeFeatureLayer(wet_line,"wet_line_lyr")

        fields = ['QAQC_CODE','ATTRIBUTE']

        edit = arcpy.da.Editor(input_gdb)
        edit.startEditing()
        edit.startOperation()
        with arcpy.da.UpdateCursor("wet_line_lyr",fields) as cursor:
            #arcpy.AddMessage("update cursor processing...")
            for row in cursor:

                if row[0] is None:
                    row[0] = "NNNNNN"
                
                row[0] = "N"+row[0][1:]
                newCode = "N"+row[0][1:]
                #arcpy.AddMessage("newCode = {x3}".format(x3=newCode))
                
                #arcpy.AddMessage("updated row")

                if row[1] is not None:
                    row[1] = row[1].strip(" ")
                cursor.updateRow(row)
                
        edit.stopOperation()
        edit.stopEditing(True)
        
        goodCodeList = []
        badCodeList = []

        list_fields = ['ATTRIBUTE','QAQC_CODE']
        #arcpy.AddMessage("begin ATTRIBUTE cursor")
        with arcpy.da.SearchCursor(wet_line,list_fields) as cursor:
            for row in cursor:
                code = str(row[0])
                #arcpy.AddMessage(code)
                incorrectWetlandsCodes_line.codeCheck(code,goodCodeList,badCodeList)


        if len(badCodeList) > 0:
            arcpy.management.MakeFeatureLayer(wet_line,"wet_line_lyr")
            #qry = """ATTRIBUTE IN {0}""".format(str(tuple(badCodeList)))
            if len(badCodeList) == 1:
                qry = """ATTRIBUTE IN {0})""".format(str(tuple(badCodeList))[:-2])
            else:
                qry = """ATTRIBUTE IN {0}""".format(str(tuple(badCodeList)))
            #arcpy.AddMessage(qry)
            #arcpy.AddMessage("bad codes: {x1}".format(x1=badCodeList))
        
            arcpy.management.SelectLayerByAttribute("wet_line_lyr","NEW_SELECTION",qry)
            arcpy.management.SelectLayerByAttribute("wet_line_lyr","ADD_TO_SELECTION","ATTRIBUTE IS NULL")

            fields = ['QAQC_CODE']
            edit = arcpy.da.Editor(input_gdb)
            edit.startEditing()
            edit.startOperation()
            with arcpy.da.UpdateCursor("wet_line_lyr",fields) as cursor:
                #arcpy.AddMessage("update cursor processing...")
                for row in cursor:
                    if row[0] is None:
                        row[0] = "NNNNNN"

                    row[0] = "C"+row[0][1:]

                    cursor.updateRow(row)
                    #arcpy.AddMessage("updated row")

            edit.stopOperation()
            edit.stopEditing(True)
            
            arcpy.AddMessage("Incorrect Codes (linear): {bCL}".format(bCL=badCodeList))

        return

    @staticmethod
    def codeCheck(code,goodCodeList,badCodeList):

        '''
        Early Checks
        Check if the code has already been determined to be good, cut down on processing time
        Check if code is long enough
        Check if code has invalid characters
        '''
        if code in goodCodeList:
            #print("good, already checked this one")
            return 'good'
        if code in badCodeList:
            #print("no good, you already checked this one")
            badCodeList.append(code)
            return code

        if code is None:
            badCodeList.append(code)
            return code

        if len(code)<5:
            badCodeList.append(code)
            return code

        if not code.isalnum():
            badCodeList.append(code)
            return code

        '''
        System/Subsystem Checks
        '''

        systemList = ['R','E']
        subSystemList = ['E2','R1','R2','R3','R4']

        if code[:2] not in subSystemList:
            #print("no good - system or subsystem")
            badCodeList.append(code)
            return code
        
        '''
        Class Checks
        '''
        
        classDict = {
        'E2': ['SB','US'],
        'R1': ['RB','SB','UB'],
        'R2': ['UB'],
        'R3': ['RB','UB'],
        'R4': ['SB']
        }

        if code[2:4] not in classDict[code[:2]]:
            #print("no good - class")
            badCodeList.append(code)
            return code
        
        '''
        Subclass Checks
        '''
        
        subclassDict = {
            'SB':['1','2','3','4','5','6'],
            'RB':['1','2'],
            'UB':['1','2','3','4'],
            'US':['1','2','3','4']
            }

        subclassExclusions = {
            'E2AB': ['2'],
            'E2RF': ['1'],
            'E2SB': ['7'],
            'E2US': ['5'],
            'R1EM': ['1','5'],
            'R2EM': ['1','5'],
            'R3UB': ['4'],
            'R3EM': ['1','5'],
            'R4EM': ['1','5'],
            }


        if code[4].isnumeric() and code[4] not in subclassDict[code[2:4]]:
            #print("no good - subclass")
            badCodeList.append(code)
            return code

        if code[4].isnumeric() and code[4] in subclassExclusions[code[:4]]:
            #print("no good, subclass excluded")
            badCodeList.append(code)
            return code
    

        '''
        Water Regime Checks
        '''
        
        waterRegimeDict = {
            'R1RB': ['T','V'],
            'R3RB': ['F','G','H'],
            'R1UB': ['T','V'],
            'R2UB': ['F','G','H'],
            'R3UB': ['F','G','H'],
            'R1SB': ['Q'],
            'R4SB': ['A','C','J'],
            'E2SB': ['M','N','P'],
            'E2US': ['M','N','P']
            }

        waterRegimeExclusions = {
            'R1AB1': ['Q'],
            'R1AB2': ['Q'],
            'R2AB1': ['C'],
            'R2AB2': ['C'],
            'R2US1': ['E'],
            'R2US2': ['E'],
            'R2US3': ['E'],
            'R2US4': ['A','C','J'],
            'R2US5': ['E'],
            'R3AB1': ['C'],
            'R3AB2': ['C'],
            'R3US1': ['E'],
            'R3US2': ['E'],
            'R3US3': ['E'],
            'R3US4': ['A','C','J'],
            'R3US5': ['E'],
            'R4SB6': ['A','J'],
            }
        if code[4].isnumeric() and code[5] not in waterRegimeDict[code[:4]]:
            #print("no good - water regime")
            badCodeList.append(code)
            return code
        elif code[4].isalpha() and code[4] not in waterRegimeDict[code[:4]]:
            #print("no good - water regime")
            badCodeList.append(code)
            return code

        if code[4].isnumeric() and code[:5] in waterRegimeExclusions:
            if code[5] in waterRegimeExclusions[code[:5]]:
                #print("no good - water regime excluded")
                badCodeList.append(code)
                return code
        
        '''
        Special Modifier Checks
        '''
        
        modifierDict = {
        'sm': ['b','d','f','m','h','r','s','x'],
        'wcHS': ['1','2','3','4','5','6','0'],
        'wcpH': ['a','t','i'],
        's': ['g','n']
        }

        #illegal water regime and modifier combos#
        if 'Kh' in code:
            badCodeList.append(code)
            return code

        if code[4].isnumeric():
            #print(len(code))
            #print("water regime is: {x}".format(x=code[5]))
            #print("modifiers are: {x}".format(x=code[6:]))
            modifiers = code[6:]
            if len(modifiers)==1:
                if modifiers[0] in modifierDict['sm'] or modifiers[0] in modifierDict['wcHS'] or modifiers[0] in modifierDict['wcpH'] or modifiers in modifierDict['s']:
                    #print('good')
                    #arcpy.AddMessage('good')
                    pass
                else:
                    #print('no good - special modifiers')
                    badCodeList.append(code)
                    return code
                
            elif len(modifiers)==2:
                if modifiers[0] in modifierDict['sm']:
                    if modifiers[1] in modifierDict['wcHS'] or modifiers[1] in modifierDict['wcpH'] or modifiers[1] in modifierDict['s']:
                        #print('good')
                        #arcpy.AddMessage('good')
                        pass
                    else:
                        #print('no good - special modifiers')
                        badCodeList.append(code)
                        return code
                elif modifiers[0] in modifierDict['wcHS']:
                    if modifiers[1] in modifierDict['wcpH'] or modifiers[1] in modifierDict['s']:
                        #print('good')
                        #arcpy.AddMessage('good')
                        pass
                    else:
                        #print('no good - special modifiers')
                        badCodeList.append(code)
                        return code
                elif modifiers[0] in modifierDict['wcpH']:
                    if modifiers[1] in modifierDict['s']:
                        #print('good')
                        #arcpy.AddMessage('good')
                        pass
                    else:
                        #print('no good - special modifiers')
                        badCodeList.append(code)
                        return code
                else:
                    #print('no good - special modifiers')
                    badCodeList.append(code)
                    return code
                
            elif len(modifiers)==3:
                if modifiers[0] in modifierDict['sm']:
                    if modifiers[1] in modifierDict['wcHS']:
                        if modifiers[2] in modifierDict['wcpH'] or modifiers[2] in modifierDict['s']:
                            #print('good')
                            #arcpy.AddMessage('good')
                            pass
                        else:
                            #print('no good - special modifiers')
                            badCodeList.append(code)
                            return code
                    elif modifiers[1] in modifierDict['wcpH']:
                        if modifiers[2] in modifierDict['s']:
                            #print('good')
                            #arcpy.AddMessage('good')
                            pass
                        else:
                            #print('no good - special modifiers')
                            badCodeList.append(code)
                            return code
                elif modifiers[0] in modifierDict['wcHS'] and modifiers[1] in modifierDict['wcpH'] and modifiers[2] in modifierDict['s']:
                    #print('good')
                    #arcpy.AddMessage('good')
                    pass
                else:
                    #print('no good - special modifiers')
                    badCodeList.append(code)
                    return code
                        
            elif len(modifiers)==4:
                if modifiers[0] in modifierDict['sm'] and modifiers[1] in modifierDict['wcHS'] and modifiers[2] in modifierDict['wcpH'] and modifiers[3] in modifierDict['s']:
                    #print('good')
                    #arcpy.AddMessage('good')
                    pass
                else:
                    #print('no good - special modifiers')
                    badCodeList.append(code)
                    return code
            elif len(modifiers)>4:
                #print('no good - special modifiers')
                badCodeList.append(code)
                return code

        else:
            #print(len(code))
            #print("water regime is: {x}".format(x=code[4]))
            #print("modifiers are: {x}".format(x=code[5:]))
            modifiers = code[5:]
            
            if len(modifiers)==1:
                if modifiers[0] in modifierDict['sm'] or modifiers[0] in modifierDict['wcHS'] or modifiers[0] in modifierDict['wcpH'] or modifiers in modifierDict['s']:
                    #print('good')
                    #arcpy.AddMessage('good')
                    pass
                else:
                    #print('no good - special modifiers')
                    badCodeList.append(code)
                    return code
            elif len(modifiers)==2:
                if modifiers[0] in modifierDict['sm']:
                    if modifiers[1] in modifierDict['wcHS'] or modifiers[1] in modifierDict['wcpH'] or modifiers[1] in modifierDict['s']:
                        #print('good')
                        #arcpy.AddMessage('good')
                        pass
                    else:
                        #print('no good - special modifiers')
                        badCodeList.append(code)
                        return code
                elif modifiers[0] in modifierDict['wcHS']:
                    if modifiers[1] in modifierDict['wcpH'] or modifiers[1] in modifierDict['s']:
                        #print('good')
                        #arcpy.AddMessage('good')
                        pass
                    else:
                        #print('no good - special modifiers')
                        badCodeList.append(code)
                        return code
                elif modifiers[0] in modifierDict['wcpH']:
                    if modifiers[1] in modifierDict['s']:
                        #print('good')
                        #arcpy.AddMessage('good')
                        pass
                    else:
                        #print('no good - special modifiers')
                        badCodeList.append(code)
                        return code
                else:
                    #print('no good - special modifiers')
                    badCodeList.append(code)
                    return code
                
            elif len(modifiers)==3:
                if modifiers[0] in modifierDict['sm']:
                    if modifiers[1] in modifierDict['wcHS']:
                        if modifiers[2] in modifierDict['wcpH'] or modifiers[2] in modifierDict['s']:
                            #print('good')
                            #arcpy.AddMessage('good')
                            pass
                        else:
                            #print('no good - special modifiers')
                            badCodeList.append(code)
                            return code
                    elif modifiers[1] in modifierDict['wcpH']:
                        if modifiers[2] in modifierDict['s']:
                            #print('good')
                            #arcpy.AddMessage('good')
                            pass
                        else:
                            #print('no good - special modifiers')
                            badCodeList.append(code)
                            return code
                elif modifiers[0] in modifierDict['wcHS'] and modifiers[1] in modifierDict['wcpH'] and modifiers[2] in modifierDict['s']:
                    #print('good')
                    #arcpy.AddMessage('good')
                    pass
                else:
                    #print('no good - special modifiers')
                    badCodeList.append(code)
                    return code
                        
            elif len(modifiers)==4:
                if modifiers[0] in modifierDict['sm'] and modifiers[1] in modifierDict['wcHS'] and modifiers[2] in modifierDict['wcpH'] and modifiers[3] in modifierDict['s']:
                    #print('good')
                    #arcpy.AddMessage('good')
                    pass
                else:
                    #print('no good - special modifiers')
                    badCodeList.append(code)
                    return code
            elif len(modifiers)>4:
                #print('no good - special modifiers')
                badCodeList.append(code)
                return code
        

            goodCodeList.append(code)

            #print('good, i guess')
            return 'good'



class linearOverlaps(object):
    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "Overlapping Wetlands - Linear"
        self.description = "This tool identifies overlapping wetland lines, and changes character 3 of QAQC_Code = 'O'. \n Linear wetland features in the National Wetlands Inventory are not allowed to overlap, should be reviewed, and correct. The tool exports a new feature class named 'Overlapping Wetlands' to the user's geodatabase to help identify the problematic polygons."
         

        self.canRunInBackground = False
        self.category = "Individual Linear Tools"

    
    def getParameterInfo(self):
        """Define parameter definitions"""
        # params [0]
        input_gdb = arcpy.Parameter(name = "input_gdb",
                                   displayName = "Input Geodatabase",
                                   datatype = "DEWorkspace",
                                   parameterType = "Required", #Required|Optional|Derived
                                   direction = "Input", #Input|Output
                                   )

        work_area = arcpy.Parameter(name = "work_area",
                                     displayName = "Work Area",
                                     datatype = "String",
                                     parameterType = "Required", #Required|Optional|Derived
                                     direction = "Input", #Input|Output
                                     )
        work_area.value = "CONUS"
        work_area.filter.type = "ValueList"
        work_area.filter.list = ["CONUS","AK","HI","PRVI","PacTrust"]

        #params = None
        
        return [input_gdb, work_area]

    def isLicensed(self):
        """Set whether tool is licensed to execute."""
        return True

    def updateParameters(self, parameters):
        """Modify the values and properties of parameters before internal
        validation is performed.  This method is called whenever a parameter
        has been changed."""

        return

    def updateMessages(self, parameters):
        """Modify the messages created by internal validation for each tool
        parameter.  This method is called after internal validation."""
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        wet_line = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_line".format(work_area=work_area))
        wet_projects = os.path.join(input_gdb,"{work_area}_projects".format(work_area = work_area),"{work_area}_wet_projects".format(work_area=work_area))
        if not arcpy.Exists(wet_line):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_line.".format(work_area=work_area))
        if not arcpy.Exists(wet_projects):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_projects.".format(work_area=work_area))

        return

    def execute(self, parameters, messages):
        """The source code of the tool."""

        arcpy.AddMessage("Overlapping Wetlands Linear Version 1.4.3")
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)

        arcpy.management.Compact(input_gdb)

        arcpy.AddMessage("Begin Overlapping Linear Wetlands tool")
        linearOverlaps.overlappingWetlands(input_gdb,work_area)

        return

    @staticmethod
    def overlappingWetlands(arg1,arg2):
        input_gdb = str(arg1)
        work_area = str(arg2)
        wet_line = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_line".format(work_area=work_area))
        wet_poly_topology = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wetlands_Topology".format(work_area=work_area))
        wet_projects = os.path.join(input_gdb,"{work_area}_projects".format(work_area = work_area),"{work_area}_wet_projects".format(work_area=work_area))
        qaqc_code = "'NNNNN'"
        #arcpy.management.Compact(input_gdb)

        #arcpy.AddMessage("{},{},{},{}".format(input_gdb,wet_poly,wet_poly_topology,wet_projects))
        
        #validate Topology
        arcpy.ValidateTopology_management(wet_poly_topology)

        #export topology errors
        arcpy.management.ExportTopologyErrors(wet_poly_topology, os.path.join(input_gdb, "{work_area}_wetlands".format(work_area = work_area)),'Overlapping_Linear_Wetlands')

        #delete line & point error classes
        point_topo = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"Overlapping_Linear_Wetlands_point")
        line_topo = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"Overlapping_Linear_Wetlands_line")
        poly_topo = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"Overlapping_Linear_Wetlands_poly")
        if arcpy.Exists(point_topo):
            arcpy.AddMessage("deleting point topo errors")
            arcpy.management.Delete(point_topo)
        if arcpy.Exists(poly_topo):
            arcpy.AddMessage("deleting polygonal topo errors")
            arcpy.management.Delete(poly_topo)


        arcpy.management.MakeFeatureLayer(wet_line,"wet_line_lyr")
        arcpy.management.MakeFeatureLayer(line_topo,"line_topo_lyr")
        arcpy.management.SelectLayerByAttribute("line_topo_lyr", "NEW_SELECTION", "RuleDescription = 'Must Not Overlap'")
        arcpy.management.SelectLayerByLocation("wet_line_lyr",'INTERSECT',line_topo)

        #arcpy.AddMessage("begin cursor")
        fields = ['QAQC_CODE']

        #with arcpy.da.Editor(input_gdb) as edit:
        edit = arcpy.da.Editor(input_gdb)
        edit.startEditing()
        edit.startOperation()
        with arcpy.da.UpdateCursor("wet_line_lyr",fields) as cursor:
            #arcpy.AddMessage("update cursor processing...")
            for row in cursor:

                if row[0] is None:
                    row[0] = "NNNNN"
                
                row[0] = row[0][:3]+"O"+row[0][4]
                newCode = row[0][:3]+"O"+row[0][4]
                #arcpy.AddMessage("newCode = {x3}".format(x3=newCode))
                cursor.updateRow(row)
                #arcpy.AddMessage("updated row")
        edit.stopOperation()
        edit.stopEditing(True)
                

    
                                        
        return


    
class linearPolyOverlaps_line(object):
    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "Linear and Polygonal Overlaps - Linear"
        self.description = "This tool compares the NWI wet_poly and wet_line feature classes to identify line features that overlap polygons. Two error codes are associated with this tool - an error code 'R' indicates the linear feature is connetion two riverine polygons, and in the overwhelming majority of cases should likely be a polygonal feature. An error code 'W' indicates that the linear feature overlaps a polygonal feature that is not allowed to have overlapping linear features (all codes in these classes: RB, UB, AB, RF, SB, RS, US). Overlapping features should be edited or clipped to be removed, of justification would need to be added to the QAQC table."
        self.canRunInBackground = False
        self.category = "Individual Linear Tools"

    
    def getParameterInfo(self):
        """Define parameter definitions"""
        # params [0]
        input_gdb = arcpy.Parameter(name = "input_gdb",
                                   displayName = "Input Geodatabase",
                                   datatype = "DEWorkspace",
                                   parameterType = "Required", #Required|Optional|Derived
                                   direction = "Input", #Input|Output
                                   )

        work_area = arcpy.Parameter(name = "work_area",
                                     displayName = "Work Area",
                                     datatype = "String",
                                     parameterType = "Required", #Required|Optional|Derived
                                     direction = "Input", #Input|Output
                                     )
        work_area.value = "CONUS"
        work_area.filter.type = "ValueList"
        work_area.filter.list = ["CONUS","AK","HI","PRVI","PacTrust"]

        #params = None
        
        return [input_gdb, work_area]
    
    def isLicensed(self):
        """Set whether tool is licensed to execute."""
        return True

    def updateParameters(self, parameters):
        """Modify the values and properties of parameters before internal
        validation is performed.  This method is called whenever a parameter
        has been changed."""

        return

    def updateMessages(self, parameters):
        """Modify the messages created by internal validation for each tool
        parameter.  This method is called after internal validation."""
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        wet_line = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_line".format(work_area=work_area))
        wet_projects = os.path.join(input_gdb,"{work_area}_projects".format(work_area = work_area),"{work_area}_wet_projects".format(work_area=work_area))
        if not arcpy.Exists(wet_poly):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_poly.".format(work_area=work_area))
        if not arcpy.Exists(wet_line):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_line.".format(work_area=work_area))
        if not arcpy.Exists(wet_projects):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_projects.".format(work_area=work_area))

        return

    def execute(self, parameters, messages):
        """The source code of the tool."""

        arcpy.AddMessage("Linear and Polygonal Overlaps Version 1.4.3")
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        #wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))

        arcpy.management.Compact(input_gdb)

        arcpy.AddMessage("Begin Linear and Polygonal Overlapping Wetlands tool")
        linearPolyOverlaps_line.linearPolyOverlaps_tool(input_gdb,work_area)
        return

    @staticmethod
    def linearPolyOverlaps_tool(arg1,arg2):
        input_gdb = str(arg1)
        work_area = str(arg2)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        wet_line = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_line".format(work_area=work_area))

        qaqc_code = "'NNNNN'"
        
        #arcpy.CalculateField_management(wet_poly,"ACRES","!shape.area@acres!","PYTHON","")
        
        fields = ['ATTRIBUTE','QAQC_CODE','SHAPE@']
        #arcpy.AddMessage("begin cursor")
        edit = arcpy.da.Editor(input_gdb)
        edit.startEditing()
        edit.startOperation()

        
        
        with arcpy.da.UpdateCursor(wet_line,fields) as cursor:
            #arcpy.AddMessage("update cursor processing...")
            for row in cursor:
                #arcpy.AddMessage("Attribute: {x1}, QAQC Code: {x2}".format(x1=row[0],x2=row[1]))

                #select wet_poly that NLH aren't allowed to overlap
                arcpy.management.MakeFeatureLayer(wet_poly,"wet_poly_lyr")
                arcpy.management.SelectLayerByAttribute("wet_poly_lyr", "NEW_SELECTION", "ATTRIBUTE LIKE '_UB%' OR ATTRIBUTE LIKE '__UB%' OR ATTRIBUTE LIKE '_RB%' OR ATTRIBUTE LIKE '__RB%' OR ATTRIBUTE LIKE '_AB%' OR ATTRIBUTE LIKE '__AB%' OR ATTRIBUTE LIKE '__RF%' OR ATTRIBUTE LIKE '__SB%' OR ATTRIBUTE LIKE '__RS%' OR ATTRIBUTE LIKE '__US%'", None)

                #arcpy.management.SelectLayerByAttribute("wet_poly_lyr", 'NEW_SELECTION',"ATTRIBUTE = '{x}'".format(x=row[0]))
                arcpy.management.SelectLayerByLocation("wet_poly_lyr", 'CROSSED_BY_THE_OUTLINE_OF',row[2],'', 'SUBSET_SELECTION')
                
                if row[1] is None:
                    row[1] = "NNNNN"
                
                overlap_count = arcpy.management.GetCount("wet_poly_lyr").getOutput(0)
                if int(overlap_count) > 0:
                    #arcpy.AddMessage(">1 overlap found; assigning code W")
                    row[1] = row[1][:3]+"W"+row[1][4:]
                    newCode = row[1][:3]+"W"+row[1][4:]
                    #arcpy.AddMessage("newCode = {x3}".format(x3=newCode))
                    cursor.updateRow(row)
                    #arcpy.AddMessage("updated row")

                #select riverine wet_poly
                #arcpy.management.MakeFeatureLayer(wet_poly,"wet_poly_lyr")
                arcpy.management.SelectLayerByAttribute("wet_poly_lyr", "NEW_SELECTION", "ATTRIBUTE LIKE 'R%'", None)

                #arcpy.management.SelectLayerByAttribute("wet_poly_lyr", 'NEW_SELECTION',"ATTRIBUTE = '{x}'".format(x=row[0]))
                arcpy.management.SelectLayerByLocation("wet_poly_lyr", 'CROSSED_BY_THE_OUTLINE_OF',row[2],'', 'SUBSET_SELECTION')
                
                if row[1] is None:
                    row[1] = "NNNNN"
                
                overlap_count = arcpy.management.GetCount("wet_poly_lyr").getOutput(0)
                if int(overlap_count) > 1:
                    #arcpy.AddMessage("more than 1 overlaps found; assigning code R")
                    row[1] = row[1][:3]+"R"+row[1][4:]
                    newCode = row[1][:3]+"R"+row[1][4:]
                    #arcpy.AddMessage("newCode = {x3}".format(x3=newCode))
                    cursor.updateRow(row)
                    #arcpy.AddMessage("updated row")
                
        edit.stopOperation()
        edit.stopEditing(True)
    
        return

class linearGaps_line(object):
    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "Gaps - Linear"
        self.description = "This tool compares the NWI wet_poly and wet_line feature classes to identify line features that have gaps between them and other features. An error code 'G' indicates there is a possible gap between the line and another feature. The gap should be investigated to see if it is intended and real."
        self.canRunInBackground = False
        self.category = "Individual Linear Tools"

    
    def getParameterInfo(self):
        """Define parameter definitions"""
        # params [0]
        input_gdb = arcpy.Parameter(name = "input_gdb",
                                   displayName = "Input Geodatabase",
                                   datatype = "DEWorkspace",
                                   parameterType = "Required", #Required|Optional|Derived
                                   direction = "Input", #Input|Output
                                   )

        work_area = arcpy.Parameter(name = "work_area",
                                     displayName = "Work Area",
                                     datatype = "String",
                                     parameterType = "Required", #Required|Optional|Derived
                                     direction = "Input", #Input|Output
                                     )
        work_area.value = "CONUS"
        work_area.filter.type = "ValueList"
        work_area.filter.list = ["CONUS","AK","HI","PRVI","PacTrust"]

        #params = None
        
        return [input_gdb, work_area]
    
    def isLicensed(self):
        """Set whether tool is licensed to execute."""
        return True

    def updateParameters(self, parameters):
        """Modify the values and properties of parameters before internal
        validation is performed.  This method is called whenever a parameter
        has been changed."""

        return

    def updateMessages(self, parameters):
        """Modify the messages created by internal validation for each tool
        parameter.  This method is called after internal validation."""
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_line = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_line".format(work_area=work_area))
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        wet_projects = os.path.join(input_gdb,"{work_area}_projects".format(work_area = work_area),"{work_area}_wet_projects".format(work_area=work_area))
        if not arcpy.Exists(wet_poly):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_poly.".format(work_area=work_area))
        if not arcpy.Exists(wet_line):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_line.".format(work_area=work_area))
        if not arcpy.Exists(wet_projects):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_projects.".format(work_area=work_area))

        return

    def execute(self, parameters, messages):
        """The source code of the tool."""

        arcpy.AddMessage("Linear Gaps Version 1.4.3")
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        #wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))

        arcpy.management.Compact(input_gdb)

        arcpy.AddMessage("Begin Linear Gaps tool")
        linearGaps_line.linear_gaps_tool(input_gdb,work_area)
        return

    @staticmethod
    def linear_gaps_tool(arg1,arg2):
        input_gdb = str(arg1)
        work_area = str(arg2)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        wet_line = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_line".format(work_area=work_area))

        qaqc_code = "'NNNNN'"
        
        #arcpy.CalculateField_management(wet_poly,"ACRES","!shape.area@acres!","PYTHON","")
        
        fields = ['QAQC_CODE','SHAPE@']
        #arcpy.AddMessage("begin cursor")
        edit = arcpy.da.Editor(input_gdb)
        edit.startEditing()
        edit.startOperation()

        with arcpy.da.UpdateCursor(wet_line,fields) as cursor:
            #arcpy.AddMessage("update cursor processing...")
            for row in cursor:
                #arcpy.AddMessage("Attribute: {x1}, QAQC Code: {x2}".format(x1=row[0],x2=row[1]))

                arcpy.management.MakeFeatureLayer(wet_poly,"wet_poly_lyr")
                arcpy.management.MakeFeatureLayer(wet_line,"wet_line_lyr")

                #select all polys within small distance of @SHAPE
                arcpy.management.SelectLayerByLocation("wet_poly_lyr", "WITHIN_A_DISTANCE", row[1], "5 Meters", "NEW_SELECTION", "NOT_INVERT")
                #change this to intersect and remove those from selection that intersect
                arcpy.management.SelectLayerByLocation("wet_poly_lyr", "INTERSECT", row[1], None, "REMOVE_FROM_SELECTION", "NOT_INVERT")
                
                if row[0] is None:
                    row[0] = "NNNNN"
                
                gap_count_poly = arcpy.management.GetCount("wet_poly_lyr").getOutput(0)
                if int(gap_count_poly) > 0:
                    #arcpy.AddMessage("at least 1 gaps found; assigning code G")
                    row[0] = row[0][:4]+"G"
                    newCode = row[0][:4]+"G"
                    #arcpy.AddMessage("newCode = {x3}".format(x3=newCode))
                    cursor.updateRow(row)
                    #arcpy.AddMessage("updated row")

                #select all lines within small distance of @SHAPE
                arcpy.management.SelectLayerByLocation("wet_line_lyr", "WITHIN_A_DISTANCE", row[1], "5 Meters", "NEW_SELECTION", "NOT_INVERT")
                #change this to intersect and remove those from selection that intersect
                arcpy.management.SelectLayerByLocation("wet_line_lyr", "INTERSECT", row[1], None, "REMOVE_FROM_SELECTION", "NOT_INVERT")
                
                if row[0] is None:
                    row[0] = "NNNNN"
                
                gap_count_line = arcpy.management.GetCount("wet_poly_lyr").getOutput(0)
                if int(gap_count_line) > 0:
                    #arcpy.AddMessage("at least 1 gaps found; assigning code G")
                    row[0] = row[0][:4]+"G"
                    newCode = row[0][:4]+"G"
                    #arcpy.AddMessage("newCode = {x3}".format(x3=newCode))
                    cursor.updateRow(row)
                    #arcpy.AddMessage("updated row")
                
        edit.stopOperation()
        edit.stopEditing(True)
    
        return

class linearDataPrep(object):
    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "Data Model Prep - Linear"
        self.description = "This tool dissolves the linear feature class on ATTRIBUTE and then explodes the features to ensure there are no multipart features. This process prepares the linear feature class to be in the NWI wet line data model."
        self.canRunInBackground = False
        self.category = "Individual Linear Tools"

    
    def getParameterInfo(self):
        """Define parameter definitions"""
        # params [0]
        input_gdb = arcpy.Parameter(name = "input_gdb",
                                   displayName = "Input Geodatabase",
                                   datatype = "DEWorkspace",
                                   parameterType = "Required", #Required|Optional|Derived
                                   direction = "Input", #Input|Output
                                   )

        work_area = arcpy.Parameter(name = "work_area",
                                     displayName = "Work Area",
                                     datatype = "String",
                                     parameterType = "Required", #Required|Optional|Derived
                                     direction = "Input", #Input|Output
                                     )
        work_area.value = "CONUS"
        work_area.filter.type = "ValueList"
        work_area.filter.list = ["CONUS","AK","HI","PRVI","PacTrust"]

        #params = None
        
        return [input_gdb, work_area]
    
    def isLicensed(self):
        """Set whether tool is licensed to execute."""
        return True

    def updateParameters(self, parameters):
        """Modify the values and properties of parameters before internal
        validation is performed.  This method is called whenever a parameter
        has been changed."""

        return

    def updateMessages(self, parameters):
        """Modify the messages created by internal validation for each tool
        parameter.  This method is called after internal validation."""
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        wet_line = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_line".format(work_area=work_area))
        wet_projects = os.path.join(input_gdb,"{work_area}_projects".format(work_area = work_area),"{work_area}_wet_projects".format(work_area=work_area))
        if not arcpy.Exists(wet_poly):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_poly.".format(work_area=work_area))
        if not arcpy.Exists(wet_line):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_line.".format(work_area=work_area))
        if not arcpy.Exists(wet_projects):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_projects.".format(work_area=work_area))

        return

    def execute(self, parameters, messages):
        """The source code of the tool."""

        arcpy.AddMessage("Linear Data Model Prep Version 1.4.3")
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        #wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))

        arcpy.management.Compact(input_gdb)

        arcpy.AddMessage("Begin Linear Data Model Prep tool")
        linearDataPrep.linear_data_prep_tool(input_gdb,work_area)
        return

    @staticmethod
    def linear_data_prep_tool(arg1,arg2):
        input_gdb = str(arg1)
        work_area = str(arg2)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        wet_line = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_line".format(work_area=work_area))

        #arcpy.CalculateField_management(wet_poly,"ACRES","!shape.area@acres!","PYTHON","")

        #wetLineDissolve = r"memory\wet_line_dissolve"
        wetLineDissolve = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_line_dissolve".format(work_area=work_area))
        arcpy.management.Dissolve(wet_line, wetLineDissolve, "ATTRIBUTE", "QAQC_CODE FIRST;WETLAND_TYPE FIRST", "SINGLE_PART", "DISSOLVE_LINES")
        #wetLineDissolvePoints = r"memory\wet_line_dissolve_points"
        wetLineDissolvePoints = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_line_dissolve_points".format(work_area=work_area))
        arcpy.analysis.Intersect(wetLineDissolve, wetLineDissolvePoints, "ALL", None, "POINT")
        #wetLineDissolveSplit = r"memory\wet_line_dissolve_split"
        wetLineDissolveSplit = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_line_dissolve_split".format(work_area=work_area))
        arcpy.management.SplitLineAtPoint(wetLineDissolve, wetLineDissolvePoints, wetLineDissolveSplit, "0.25 Meters")

        wet_line_backup = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_line_backup".format(work_area=work_area))
        if arcpy.Exists(wet_line_backup):
            arcpy.management.Delete(wet_line_backup)
        arcpy.management.CopyFeatures(wet_line,wet_line_backup)
        #arcpy.management.Copy(wet_line,wet_line_backup)

        #Truncate Table doesn't work on feature classes in a feature dataset
        #arcpy.TruncateTable_management(wet_line)
        arcpy.management.DeleteFeatures(wet_line)

        fieldMappings = arcpy.FieldMappings()
        fieldMappings.addTable(wet_line)

        att_fm = arcpy.FieldMap()
        att_fm.addInputField(wetLineDissolveSplit,"ATTRIBUTE")
        attribute = att_fm.outputField
        attribute.name,attribute.aliasName,attribute.type = "ATTRIBUTE","ATTRIBUTE","TEXT"
        att_fm.outputField = attribute
        fieldMappings.addFieldMap(att_fm)

        qaqc_fm = arcpy.FieldMap()
        qaqc_fm.addInputField(wetLineDissolveSplit,"FIRST_QAQC_CODE")
        qaqccode = att_fm.outputField
        qaqccode.name,qaqccode.aliasName,qaqccode.type = "QAQC_CODE","QAQC_CODE","TEXT"
        qaqc_fm.outputField = qaqccode
        fieldMappings.addFieldMap(qaqc_fm)
        
        wetlandtype_fm = arcpy.FieldMap()
        wetlandtype_fm.addInputField(wetLineDissolveSplit,"FIRST_WETLAND_TYPE")
        wetlandtype = wetlandtype_fm.outputField
        wetlandtype.name,wetlandtype.aliasName,wetlandtype.type = "WETLAND_TYPE","WETLAND_TYPE","TEXT"
        wetlandtype_fm.outputField = wetlandtype
        fieldMappings.addFieldMap(wetlandtype_fm)

        #now append or load the dissolved split data into the wet_line
        arcpy.management.Append(wetLineDissolveSplit, wet_line, "NO_TEST", fieldMappings)


        return


class combinedLineTools(object):
    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "Combined Wetland Linear Tools"
        self.description = "This tool runs all the linear QAQC Tools."
        self.canRunInBackground = False
        self.category = "Combined Tools"
        #self.QAQCCODERESET = QAQCcodeReset()
        #self.SLIVERWETLANDS = sliverWetlands()

    
    def getParameterInfo(self):
        """Define parameter definitions"""
        # params [0]
        input_gdb = arcpy.Parameter(name = "input_gdb",
                                   displayName = "Input Geodatabase",
                                   datatype = "DEWorkspace",
                                   parameterType = "Required", #Required|Optional|Derived
                                   direction = "Input", #Input|Output
                                   )

        work_area = arcpy.Parameter(name = "work_area",
                                     displayName = "Work Area",
                                     datatype = "String",
                                     parameterType = "Required", #Required|Optional|Derived
                                     direction = "Input", #Input|Output
                                     )

        work_area.value = "CONUS"
        work_area.filter.type = "ValueList"
        work_area.filter.list = ["CONUS","AK","HI","PRVI","PacTrust"]

        verifier_name = arcpy.Parameter(name = "verifier_name",
                                        displayName = "Verified By:",
                                        datatype = "String",
                                        parameterType = "Required",
                                        direction = "Input",
                                        )

        save_summary_table = arcpy.Parameter(name = "save_summary_table",
                                             displayName = "Save Summary Table?",
                                             datatype = "GPBoolean",
                                             parameterType = "Required",
                                             direction = "Input",
                                             )
        save_summary_table.value = True

        #params = None
        return [input_gdb,work_area,verifier_name,save_summary_table]

    def isLicensed(self):
        """Set whether tool is licensed to execute."""
        return True

    def updateParameters(self, parameters):
        """Modify the values and properties of parameters before internal
        validation is performed.  This method is called whenever a parameter
        has been changed."""

        return

    def updateMessages(self, parameters):
        """Modify the messages created by internal validation for each tool
        parameter.  This method is called after internal validation."""
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        wet_line = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_line".format(work_area=work_area))
        wet_projects = os.path.join(input_gdb,"{work_area}_projects".format(work_area = work_area),"{work_area}_wet_projects".format(work_area=work_area))
        if not arcpy.Exists(wet_poly):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_poly.".format(work_area=work_area))
        if not arcpy.Exists(wet_line):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_line.".format(work_area=work_area))
        if not arcpy.Exists(wet_projects):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_projects.".format(work_area=work_area))

        return

    def execute(self, parameters, messages):
        """The source code of the tool."""

        arcpy.AddMessage("Combined Linear Tools Version 1.4.3")
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        wet_line = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_line".format(work_area=work_area))
        wet_projects = os.path.join(input_gdb,"{work_area}_projects".format(work_area = work_area),"{work_area}_wet_projects".format(work_area=work_area))

        arcpy.management.Compact(input_gdb)

        verifier_name = str(parameters[2].value)
        save_summary_table = str(parameters[3].value)

        #arcpy.AddMessage(verifier_name)
        #arcpy.AddMessage(save_summary_table)

        #Data Prep
        linearDataPrep.linear_data_prep_tool(input_gdb,work_area)

        QAQCcodeReset_line.QAQCcodeReset_tool(input_gdb,work_area)
        wetlandTypeCalc_line.wetlandTypeCalc(input_gdb,work_area)
        create_NWIID.create_NWIID_tool(input_gdb,work_area,"purple")
        if work_area == "CONUS":
            create_NWIID.create_NWIID_tool(input_gdb,work_area,"purple")
        
        incorrectWetlandsCodes_line.incorrectCodes(input_gdb,work_area)
        linearPolyOverlaps_line.linearPolyOverlaps_tool(input_gdb,work_area)
        sliverWetlands_line.sliverWetlands(input_gdb,work_area)
        linearOverlaps.overlappingWetlands(input_gdb,work_area)
        linearGaps_line.linear_gaps_tool(input_gdb,work_area)
        
        
        QAQCSummary_poly.QAQCSummary_tool(input_gdb,work_area,verifier_name,save_summary_table)

        return

class QAQCSummary_line(object):
    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "QAQC Summary - Linear"
        self.description = "Summarizes the QAQC_CODE field into a 'QAQC_Summary' table in your wetlands file geodatabase. It also describes each error type and records who conducted the verification and when the verification was run. Comments can be added to the ‘comments’ field of the QAQC_Summary table to justify specific types of errors. These comments will only be saved if the ‘Append to History Table’ box is checked prior to running the All_QAQC_Checks model. "
         

        self.canRunInBackground = False
        self.category = "Individual Linear Tools"

    
    def getParameterInfo(self):
        """Define parameter definitions"""
        # params [0]
        input_gdb = arcpy.Parameter(name = "input_gdb",
                                   displayName = "Input Geodatabase",
                                   datatype = "DEWorkspace",
                                   parameterType = "Required", #Required|Optional|Derived
                                   direction = "Input", #Input|Output
                                   )

        work_area = arcpy.Parameter(name = "work_area",
                                     displayName = "Work Area",
                                     datatype = "String",
                                     parameterType = "Required", #Required|Optional|Derived
                                     direction = "Input", #Input|Output
                                     )
        work_area.value = "CONUS"
        work_area.filter.type = "ValueList"
        work_area.filter.list = ["CONUS","AK","HI","PRVI","PacTrust"]

        verifier_name = arcpy.Parameter(name = "verifier_name",
                                        displayName = "Verified By:",
                                        datatype = "String",
                                        parameterType = "Required",
                                        direction = "Input",
                                        )

        save_summary_table = arcpy.Parameter(name = "save_summary_table",
                                             displayName = "Save Summary Table?",
                                             datatype = "GPBoolean",
                                             parameterType = "Required",
                                             direction = "Input",
                                             )
        save_summary_table.value = True
        #params = None
        
        return [input_gdb, work_area,verifier_name,save_summary_table]

    def isLicensed(self):
        """Set whether tool is licensed to execute."""
        return True

    def updateParameters(self, parameters):
        """Modify the values and properties of parameters before internal
        validation is performed.  This method is called whenever a parameter
        has been changed."""

        return

    def updateMessages(self, parameters):
        """Modify the messages created by internal validation for each tool
        parameter.  This method is called after internal validation."""
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        wet_line = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_line".format(work_area=work_area))
        wet_projects = os.path.join(input_gdb,"{work_area}_projects".format(work_area = work_area),"{work_area}_wet_projects".format(work_area=work_area))
        if not arcpy.Exists(wet_line):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_line.".format(work_area=work_area))
        

        return

    def execute(self, parameters, messages):
        """The source code of the tool."""

        arcpy.AddMessage("QAQC Summary Version 1.4.3")
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        verifier_name = str(parameters[2].value)
        save_summary_table = str(parameters[3].value)
        #wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        
        arcpy.management.Compact(input_gdb)
        
        arcpy.AddMessage("Begin QAQC Summary tool")
        QAQCSummary_poly.QAQCSummary_tool(input_gdb,work_area,verifier_name,save_summary_table)
                        
        return

    @staticmethod
    def QAQCSummary_tool(arg1,arg2,arg3,arg4):

        input_gdb = str(arg1)
        work_area = str(arg2)
        verifier_name = str(arg3)
        save_summary_table = arg4
        #arcpy.AddMessage("arg4: {x1},type(arg4): {x2}".format(x1=arg4,x2=type(arg4)))
        #wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        wet_line = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_line".format(work_area=work_area))
        summaryTable = os.path.join(input_gdb,"QAQC_Linear_Summary")
        #arcpy.management.Compact(input_gdb)
        
        arcpy.analysis.Frequency(wet_line,summaryTable,["QAQC_CODE"])
        arcpy.management.AddField(summaryTable,"ERROR_TYPE","TEXT","","",255)
        arcpy.management.AddField(summaryTable,"VERIFIED_BY","TEXT","","",255)
        arcpy.management.AddField(summaryTable,"VERIFICATION_DATE","DATE")
        arcpy.management.AddField(summaryTable,"VERIFICATION_COMMENTS","TEXT","","",255)
        
        
        #arcpy.AddMessage("begin cursor")
        fields = ['QAQC_CODE','ERROR_TYPE']
        

        edit = arcpy.da.Editor(input_gdb)
        edit.startEditing()
        edit.startOperation()
        with arcpy.da.UpdateCursor(summaryTable,fields) as cursor:
            #arcpy.AddMessage("update cursor processing...")
            for row in cursor:
                newList = []

                if len(row[0]) == 6:
                    if row[0] == "NNNNN":
                        row[1] = "Passed Verification"
                    else:
                        printDict = {
                            'C': 'Bad Code',
                            'W': 'Overlapping Polygonal Wetland',
                            'R': 'Connecting Wetland',
                            'S': 'Sliver Wetland',
                            'O': 'Overlapping Linear Wetland',
                            'G': 'Gap Wetland',
                            ',': ', '
                            }
                        #newList = []
                        #newListString = ''
                        for i,v in enumerate(row[0]):
                            #arcpy.AddMessage(v)
                            #arcpy.AddMessage(type(newList))
                            if v != 'N':
                                newList.append(v)
                                
                            newListComma = ','.join(newList)
                            #arcpy.AddMessage('newListComma: {x}'.format(x=newListComma))
                            
                            printString = ''
                            for char in newListComma:
                                nLC = str(printDict[char])
                                printString += nLC

                            #arcpy.AddMessage('printString: {y}'.format(y=printString))
                            row[1] = printString
                cursor.updateRow(row)
                #arcpy.AddMessage("updated row")
                
        edit.stopOperation()
        edit.stopEditing(True)

        arcpy.CalculateField_management(summaryTable,"VERIFIED_BY","'"+verifier_name+"'","PYTHON","")
        #date_value = datetime.datetime.now()
        arcpy.CalculateField_management(summaryTable,"VERIFICATION_DATE","datetime.datetime.now()","PYTHON","")

        history_table = os.path.join(input_gdb,"QAQC_History")
        if arcpy.Exists(history_table):
            run_list = []
            with arcpy.da.SearchCursor(history_table,'RUN') as cursor:
                for row in cursor:
                    run_list.append(row[0])
            run_max = max(run_list)
            #arcpy.AddMessage("run_max: {x1}".format(x1=run_max))
        
            #arcpy.AddMessage("history table exists")

            arcpy.AddField_management(summaryTable,"RUN","SHORT")
            arcpy.CalculateField_management(summaryTable,"RUN",run_max+1)
            arcpy.management.Append(summaryTable,history_table,"TEST")
        else:
            arcpy.AddField_management(summaryTable,"RUN","SHORT")
            arcpy.CalculateField_management(summaryTable,"RUN",1)
            #arcpy.CopyFeatures_management(summaryTable,history_table)
            arcpy.conversion.TableToTable(summaryTable,input_gdb,'QAQC_History')


        return


class combinedPolyandLineTools(object):
    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "Combined Wetland Polygonal and Linear Tools"
        self.description = "This tool runs all the polygonal and linear QAQC Tools."
        self.canRunInBackground = False
        self.category = "Combined Tools"
            
    def getParameterInfo(self):
        """Define parameter definitions"""
        # params [0]
        input_gdb = arcpy.Parameter(name = "input_gdb",
                                   displayName = "Input Geodatabase",
                                   datatype = "DEWorkspace",
                                   parameterType = "Required", #Required|Optional|Derived
                                   direction = "Input", #Input|Output
                                   )

        work_area = arcpy.Parameter(name = "work_area",
                                     displayName = "Work Area",
                                     datatype = "String",
                                     parameterType = "Required", #Required|Optional|Derived
                                     direction = "Input", #Input|Output
                                     )

        work_area.value = "CONUS"
        work_area.filter.type = "ValueList"
        work_area.filter.list = ["CONUS","AK","HI","PRVI","PacTrust"]

        verifier_name = arcpy.Parameter(name = "verifier_name",
                                        displayName = "Verified By:",
                                        datatype = "String",
                                        parameterType = "Required",
                                        direction = "Input",
                                        )

        save_summary_table = arcpy.Parameter(name = "save_summary_table",
                                             displayName = "Save Summary Table?",
                                             datatype = "GPBoolean",
                                             parameterType = "Required",
                                             direction = "Input",
                                             )
        save_summary_table.value = True
        
        intermediate_data = arcpy.Parameter(name = "intermediate_data_type",
                                            displayName = "Temporary Data Type",
                                            datatype = "GPBoolean",
                                            parameterType = "Required",
                                            direction = "Input"
                                            )

        intermediate_data.value = False
        
        project_division = arcpy.Parameter(name = "project_division_number",
                                           displayName = "Number of Processing Divisions",
                                           datatype = "GPLong",
                                           parameterType = "Required",
                                           direction = "Input")

        project_division.value = 1
        project_division.filter.type = "ValueList"
        project_division.filter.list = [1,4,9,16,25]

        #params = None
        return [input_gdb,work_area,verifier_name,save_summary_table,intermediate_data,project_division]

    def isLicensed(self):
        """Set whether tool is licensed to execute."""
        return True

    def updateParameters(self, parameters):
        """Modify the values and properties of parameters before internal
        validation is performed.  This method is called whenever a parameter
        has been changed."""

        return

    def updateMessages(self, parameters):
        """Modify the messages created by internal validation for each tool
        parameter.  This method is called after internal validation."""
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        wet_line = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_line".format(work_area=work_area))
        wet_projects = os.path.join(input_gdb,"{work_area}_projects".format(work_area = work_area),"{work_area}_wet_projects".format(work_area=work_area))
        if not arcpy.Exists(wet_poly):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_poly.".format(work_area=work_area))
        if not arcpy.Exists(wet_line):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_line.".format(work_area=work_area))
        if not arcpy.Exists(wet_projects):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_projects.".format(work_area=work_area))

        return

    def execute(self, parameters, messages):
        """The source code of the tool."""

        arcpy.AddMessage("Combined Polygonal and Linear Tools Version 1.4.3")
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        wet_line = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_line".format(work_area=work_area))
        wet_projects = os.path.join(input_gdb,"{work_area}_projects".format(work_area = work_area),"{work_area}_wet_projects".format(work_area=work_area))

        arcpy.management.Compact(input_gdb)

        verifier_name = str(parameters[2].value)
        save_summary_table = str(parameters[3].value)
        intermediate_data = parameters[4].value
        project_division = parameters[5].value

        #arcpy.AddMessage(verifier_name)
        #arcpy.AddMessage(save_summary_table)

        arcpy.AddMessage("Resetting polygonal QAQC code and calculating Wetland Type")
        QAQCcodeReset_poly.QAQCcodeReset_tool(input_gdb,work_area)
        wetlandTypeCalc_poly.wetlandTypeCalc(input_gdb,work_area)

        create_NWIID.create_NWIID_tool(input_gdb,work_area,"wet")
        

        arcpy.AddMessage("Begin Incorrect Wetlands Codes polygonal tool")
        incorrectWetlandsCodes_poly.incorrectCodes(input_gdb,work_area)
        arcpy.AddMessage("Begin Adjacent Wetlands polygonal tool")
        adjacentWetlands.adjacentWetlands(input_gdb,work_area,intermediate_data)
        arcpy.AddMessage("Begin Sliver Wetlands polygonal tool")
        sliverWetlands_poly.sliverWetlands(input_gdb,work_area)
        arcpy.AddMessage("Begin Sliver Uplands polygonal tool")
        sliverUplands_poly.sliverUplands(input_gdb,work_area,intermediate_data,project_division)
        arcpy.AddMessage("Begin Lake and Pond size polygonal tool")
        lakeAndPondSize.lakeAndPondSize(input_gdb,work_area,intermediate_data)
        arcpy.AddMessage("Begin Overlapping Wetlands polygonal tool")
        overlappingWetlands_poly.overlappingWetlands(input_gdb,work_area)
        
        linear_count = arcpy.management.GetCount(wet_line).getOutput(0)
        if int(linear_count) > 0:
        
            arcpy.AddMessage("Resetting linear QAQC code and calculating Wetland Type")
            QAQCcodeReset_line.QAQCcodeReset_tool(input_gdb,work_area)
            wetlandTypeCalc_line.wetlandTypeCalc(input_gdb,work_area)
            create_NWIID.create_NWIID_tool(input_gdb,work_area,"line")
        
            
            arcpy.AddMessage("Begin Incorrect Wetlands Codes linear tool")
            incorrectWetlandsCodes_line.incorrectCodes(input_gdb,work_area)
            arcpy.AddMessage("Begin Linear and Polygonal Overlaps tool")
            linearPolyOverlaps_line.linearPolyOverlaps_tool(input_gdb,work_area)
            arcpy.AddMessage("Begin Sliver Wetlands linear tool")
            sliverWetlands_line.sliverWetlands(input_gdb,work_area)
            arcpy.AddMessage("Begin Overlapping Wetlands linear tool")
            linearOverlaps.overlappingWetlands(input_gdb,work_area)
            arcpy.AddMessage("Begin Linear Gaps tool")
            linearGaps_line.linear_gaps_tool(input_gdb,work_area)
        
        arcpy.AddMessage("Begin QAQC Summary tool")
        QAQCSummary_poly.QAQCSummary_tool(input_gdb,work_area,verifier_name,save_summary_table)

        return

'''
BEGIN RIPARIAN SECTION
'''

class QAQCcodeReset_Riparian(object):
    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "QAQC Code Reset - Riparian"
        self.description = "This tool calculates the QAQC_Code to be 'NNNNNN'. \n This erases all recorded errors in the dataset and properly attributes the field for use by all other models. "
        self.canRunInBackground = False
        self.category = "Individual Riparian Tools"

    
    def getParameterInfo(self):
        """Define parameter definitions"""
        # params [0]
        input_gdb = arcpy.Parameter(name = "input_gdb",
                                   displayName = "Input Geodatabase",
                                   datatype = "DEWorkspace",
                                   parameterType = "Required", #Required|Optional|Derived
                                   direction = "Input", #Input|Output
                                   )

        work_area = arcpy.Parameter(name = "work_area",
                                     displayName = "Work Area",
                                     datatype = "String",
                                     parameterType = "Required", #Required|Optional|Derived
                                     direction = "Input", #Input|Output
                                     )
        work_area.value = "CONUS"
        work_area.filter.type = "ValueList"
        work_area.filter.list = ["CONUS","AK","HI","PRVI","PacTrust"]

        #params = None
        return [input_gdb,work_area]

    def isLicensed(self):
        """Set whether tool is licensed to execute."""
        return True

    def updateParameters(self, parameters):
        """Modify the values and properties of parameters before internal
        validation is performed.  This method is called whenever a parameter
        has been changed."""

        return

    def updateMessages(self, parameters):
        """Modify the messages created by internal validation for each tool
        parameter.  This method is called after internal validation."""
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        wet_line = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_line".format(work_area=work_area))
        wet_projects = os.path.join(input_gdb,"{work_area}_projects".format(work_area = work_area),"{work_area}_wet_projects".format(work_area=work_area))
        rip_poly = os.path.join(input_gdb,"{work_area}_riparian".format(work_area = work_area),"{work_area}_rip_poly".format(work_area=work_area))
        if not arcpy.Exists(rip_poly):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_rip_poly.".format(work_area=work_area))

        return

    def execute(self, parameters, messages):
        """The source code of the tool."""

        arcpy.AddMessage("Riparian QAQC Reset Tool Version 1.4.3")
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        
        arcpy.management.Compact(input_gdb)

        '''
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        qaqc_code = "'NNNNNN'"
        arcpy.management.CalculateField(wet_poly,"QAQC_CODE",qaqc_code,"PYTHON_9.3")
        '''
        arcpy.AddMessage("Begin Riparian QAQC code reset tool")
        QAQCcodeReset_Riparian.QAQCcodeReset_tool(input_gdb,work_area)
        return

    @staticmethod
    def QAQCcodeReset_tool(arg1,arg2):
        input_gdb = str(arg1)
        work_area = str(arg2)
        rip_poly = os.path.join(input_gdb,"{work_area}_riparian".format(work_area = work_area),"{work_area}_rip_poly".format(work_area=work_area))

        qaqc_code = "'NNNNN'"
        arcpy.management.CalculateField(rip_poly,"QAQC_CODE",qaqc_code,"PYTHON_9.3")
        return

class riparianTypeCalc(object):
    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "Riparian Type Calculation"
        self.description = "This tool calculates the 'wetland_type' field based on the wetland code in the 'attribute' field. The 'wetland_type' field provides a general description of the wetland type and is used in the cartographic representation of the different wetland types on the Wetlands Mapper."

        self.canRunInBackground = False
        self.category = "Individual Riparian Tools"

    
    def getParameterInfo(self):
        """Define parameter definitions"""
        # params [0]
        input_gdb = arcpy.Parameter(name = "input_gdb",
                                   displayName = "Input Geodatabase",
                                   datatype = "DEWorkspace",
                                   parameterType = "Required", #Required|Optional|Derived
                                   direction = "Input", #Input|Output
                                   )

        work_area = arcpy.Parameter(name = "work_area",
                                     displayName = "Work Area",
                                     datatype = "String",
                                     parameterType = "Required", #Required|Optional|Derived
                                     direction = "Input", #Input|Output
                                     )
        work_area.value = "CONUS"
        work_area.filter.type = "ValueList"
        work_area.filter.list = ["CONUS","AK","HI","PRVI","PacTrust"]

        #params = None
        
        return [input_gdb, work_area]

    def isLicensed(self):
        """Set whether tool is licensed to execute."""
        return True

    def updateParameters(self, parameters):
        """Modify the values and properties of parameters before internal
        validation is performed.  This method is called whenever a parameter
        has been changed."""

        return

    def updateMessages(self, parameters):
        """Modify the messages created by internal validation for each tool
        parameter.  This method is called after internal validation."""

        arcpy.AddMessage("Riparian Type Calculation Version 1.4.3")
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        wet_line = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_line".format(work_area=work_area))
        wet_projects = os.path.join(input_gdb,"{work_area}_projects".format(work_area = work_area),"{work_area}_wet_projects".format(work_area=work_area))
        rip_poly = os.path.join(input_gdb,"{work_area}_riparian".format(work_area = work_area),"{work_area}_rip_poly".format(work_area=work_area))
        if not arcpy.Exists(rip_poly):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_rip_poly.".format(work_area=work_area))

        return

    def execute(self, parameters, messages):
        """The source code of the tool."""

        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        #wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))

        arcpy.management.Compact(input_gdb)

        arcpy.AddMessage("Begin Riparian Type Calculation tool")
        riparianTypeCalc.riparianTypeCalc(input_gdb,work_area)
        
        return

    @staticmethod
    def riparianTypeCalc(arg1,arg2):

        input_gdb = str(arg1)
        work_area = str(arg2)
        rip_poly = os.path.join(input_gdb,"{work_area}_riparian".format(work_area = work_area),"{work_area}_rip_poly".format(work_area=work_area))
        #try calculate field option here
        
        fields = ['ATTRIBUTE','WETLAND_TYPE']
        #arcpy.AddMessage("begin cursor")
        edit = arcpy.da.Editor(input_gdb)
        edit.startEditing()
        edit.startOperation()
        with arcpy.da.UpdateCursor(rip_poly,fields) as cursor:
            #arcpy.AddMessage("update cursor processing...")
            for row in cursor:
                #arcpy.AddMessage("row[0]: {x1}, row[1]: {x2}".format(x1=row[0],x2=row[1]))
                if row[0] is None:
                    row[1] = "NA"
                elif len(row[0])<5:
                    row[1] = "NA"
                elif row[0][:5] == "Rp1FO":
                    row[1] = "Forested/Shrub Riparian"
                elif row[0][:5] == "Rp1SS":
                    row[1] = "Forested/Shrub Riparian"
                elif row[0][:5] == "Rp2SS":
                    row[1] = "Forested/Shrub Riparian"
                elif row[0][:5] == "Rp2FO":
                    row[1] = "Forested/Shrub Riparian"
                elif row[0][:5] == "Rp1EM":
                    row[1] = "Herbaceous Riparian"
                elif row[0][:5] == "Rp2EM":
                    row[1] = "Herbaceous Riparian"
                else:
                    row[1] = "Other"

                cursor.updateRow(row)
                #arcpy.AddMessage("updated row")
        edit.stopOperation()
        edit.stopEditing(True)
                                        
        return

class sliverRiparian(object):
    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "Sliver Riparian"
        self.description = "This tool identifies riparian polygons less than 0.01 acres, and changes character 3 of QAQC_Code = 'S'. \n These wetland features fall below the minimum mapping standard for riparian data and should be reviewed. Actual wetland features flagged as sliver wetlands can be justified as correct in the comments field of the QAQC_Summary table. These comments will only be saved if the ‘Append to History Table’ box is checked prior to running the All_QAQC_Checks model."
         

        self.canRunInBackground = False
        self.category = "Individual Riparian Tools"

    
    def getParameterInfo(self):
        """Define parameter definitions"""
        # params [0]
        input_gdb = arcpy.Parameter(name = "input_gdb",
                                   displayName = "Input Geodatabase",
                                   datatype = "DEWorkspace",
                                   parameterType = "Required", #Required|Optional|Derived
                                   direction = "Input", #Input|Output
                                   )

        work_area = arcpy.Parameter(name = "work_area",
                                     displayName = "Work Area",
                                     datatype = "String",
                                     parameterType = "Required", #Required|Optional|Derived
                                     direction = "Input", #Input|Output
                                     )
        work_area.value = "CONUS"
        work_area.filter.type = "ValueList"
        work_area.filter.list = ["CONUS","AK","HI","PRVI","PacTrust"]

        #params = None
        
        return [input_gdb, work_area]
    
    def isLicensed(self):
        """Set whether tool is licensed to execute."""
        return True

    def updateParameters(self, parameters):
        """Modify the values and properties of parameters before internal
        validation is performed.  This method is called whenever a parameter
        has been changed."""

        return

    def updateMessages(self, parameters):
        """Modify the messages created by internal validation for each tool
        parameter.  This method is called after internal validation."""
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        wet_line = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_line".format(work_area=work_area))
        wet_projects = os.path.join(input_gdb,"{work_area}_projects".format(work_area = work_area),"{work_area}_wet_projects".format(work_area=work_area))
        rip_poly = os.path.join(input_gdb,"{work_area}_riparian".format(work_area = work_area),"{work_area}_rip_poly".format(work_area=work_area))
        if not arcpy.Exists(rip_poly):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_rip_poly.".format(work_area=work_area))

        return

    def execute(self, parameters, messages):
        """The source code of the tool."""

        arcpy.AddMessage("Riparian Slivers Version 1.4.3")
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        #wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))

        arcpy.management.Compact(input_gdb)

        arcpy.AddMessage("Begin Riparian Slivers tool")
        sliverRiparian.sliverRiparian_tool(input_gdb,work_area)
        return

    @staticmethod
    def sliverRiparian_tool(arg1,arg2):
        input_gdb = str(arg1)
        work_area = str(arg2)
        rip_poly = os.path.join(input_gdb,"{work_area}_riparian".format(work_area = work_area),"{work_area}_rip_poly".format(work_area=work_area))

        qaqc_code = "'NNNNN'"
        
        arcpy.CalculateField_management(rip_poly,"ACRES","!shape.area@acres!","PYTHON","")
        
        fields = ['QAQC_CODE','ACRES']
        #arcpy.AddMessage("begin cursor")
        edit = arcpy.da.Editor(input_gdb)
        edit.startEditing()
        edit.startOperation()
        with arcpy.da.UpdateCursor(rip_poly,fields) as cursor:
            #arcpy.AddMessage("update cursor processing...")
            for row in cursor:
                #arcpy.AddMessage("row[0]: {x1}, row[1]: {x2}".format(x1=row[0],x2=row[1]))
                if row[0] is None:
                    row[0] = "NNNNNN"
                if row[1] < .01:
                    #arcpy.AddMessage("row[1]<.01")
                    row[0] = row[0][:2]+"S"+row[0][3:]
                    newCode = row[0][:2]+"S"+row[0][3:]
                    #arcpy.AddMessage("newCode = {x3}".format(x3=newCode))
                    cursor.updateRow(row)
                    #arcpy.AddMessage("updated row")

        edit.stopOperation()
        edit.stopEditing(True)
    
        return


class adjacentRiparian(object):
    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "Adjacent Riparian"
        self.description = "Identifies riparian polygons that are adjacent to other riparian polygons with the same 'attribute'. Changes character 2 of QAQC_Code = 'A'."
         

        self.canRunInBackground = False
        self.category = "Individual Riparian Tools"

    
    def getParameterInfo(self):
        """Define parameter definitions"""
        # params [0]
        input_gdb = arcpy.Parameter(name = "input_gdb",
                                   displayName = "Input Geodatabase",
                                   datatype = "DEWorkspace",
                                   parameterType = "Required", #Required|Optional|Derived
                                   direction = "Input", #Input|Output
                                   )

        work_area = arcpy.Parameter(name = "work_area",
                                     displayName = "Work Area",
                                     datatype = "String",
                                     parameterType = "Required", #Required|Optional|Derived
                                     direction = "Input", #Input|Output
                                     )
        work_area.value = "CONUS"
        work_area.filter.type = "ValueList"
        work_area.filter.list = ["CONUS","AK","HI","PRVI","PacTrust"]

        #params = None
        
        return [input_gdb, work_area]

    def isLicensed(self):
        """Set whether tool is licensed to execute."""
        return True

    def updateParameters(self, parameters):
        """Modify the values and properties of parameters before internal
        validation is performed.  This method is called whenever a parameter
        has been changed."""

        return

    def updateMessages(self, parameters):
        """Modify the messages created by internal validation for each tool
        parameter.  This method is called after internal validation."""
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        wet_line = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_line".format(work_area=work_area))
        wet_projects = os.path.join(input_gdb,"{work_area}_projects".format(work_area = work_area),"{work_area}_wet_projects".format(work_area=work_area))
        rip_poly = os.path.join(input_gdb,"{work_area}_riparian".format(work_area = work_area),"{work_area}_rip_poly".format(work_area=work_area))
        if not arcpy.Exists(rip_poly):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_rip_poly.".format(work_area=work_area))

        return

    def execute(self, parameters, messages):
        """The source code of the tool."""

        arcpy.AddMessage("Riparian Adjacent Version 1.4.3")
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        #wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        #qaqc_code = "'NNNNNN'"

        arcpy.management.Compact(input_gdb)

        arcpy.AddMessage("Begin Riparian Adjacent tool")
        adjacentRiparian.adjacentRiparian(input_gdb,work_area)

        return

    @staticmethod
    def adjacentRiparian(arg1,arg2):
        arcpy.env.maintainCurveSegments = True
        input_gdb = str(arg1)
        work_area = str(arg2)
        rip_poly = os.path.join(input_gdb,"{work_area}_riparian".format(work_area = work_area),"{work_area}_rip_poly".format(work_area=work_area))

        adjacentDissolve = r"memory\dissolveAdjacent"
        #adjacentDissolve = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"adjacentDissolve")
        dissolveField = ['ATTRIBUTE']
        
        arcpy.management.Dissolve(rip_poly,adjacentDissolve,dissolveField, "", "SINGLE_PART")
        #arcpy.analysis.PairwiseDissolve(wet_poly,adjacentDissolve,dissolveField, "", "SINGLE_PART")
        arcpy.management.MakeFeatureLayer(adjacentDissolve,"adj_diss_lyr")
        arcpy.management.MakeFeatureLayer(rip_poly,"rip_poly_lyr")
        
        #arcpy.AddMessage("begin Adjacent Reset cursor")
        fields = ['QAQC_CODE']

        edit = arcpy.da.Editor(input_gdb)
        edit.startEditing()
        edit.startOperation()
        with arcpy.da.UpdateCursor("rip_poly_lyr",fields) as cursor:
            #arcpy.AddMessage("update cursor processing...")
            for row in cursor:

                if row[0] is None:
                    row[0] = "NNNNN"
                
                row[0] = row[0][:1]+"N"+row[0][2:]
                newCode = row[0][:1]+"N"+row[0][2:]
                #arcpy.AddMessage("newCode = {x3}".format(x3=newCode))
                cursor.updateRow(row)
                #arcpy.AddMessage("updated row")
        edit.stopOperation()
        edit.stopEditing(True)
        
        arcpy.management.SelectLayerByLocation("rip_poly_lyr",'ARE_IDENTICAL_TO',"adj_diss_lyr","","","INVERT")

        #begin changing adjacent features' codes to *A****
        edit = arcpy.da.Editor(input_gdb)
        edit.startEditing()
        edit.startOperation()
        with arcpy.da.UpdateCursor("rip_poly_lyr",fields) as cursor:
            #arcpy.AddMessage("update cursor processing...")
            for row in cursor:

                if row[0] is None:
                    row[0] = "NNNNN"
                
                row[0] = row[0][:1]+"A"+row[0][2:]
                newCode = row[0][:1]+"A"+row[0][2:]
                #arcpy.AddMessage("newCode = {x3}".format(x3=newCode))
                cursor.updateRow(row)
                #arcpy.AddMessage("updated row")
        edit.stopOperation()
        edit.stopEditing(True)

        mpr_lyr = arcpy.management.MakeFeatureLayer(rip_poly) 
        mprSelect = os.path.join(input_gdb,"Multipart_Riparian_Features")
        fields2 = ['OBJECTID','SHAPE@','QAQC_CODE']
        multipartripList = []
        #arcpy.AddMessage("start mp cursor")
        with arcpy.da.UpdateCursor(mpr_lyr,fields2) as cursor2:
            for row in cursor2:
                if row[1] is None:
                    cursor2.deleteRow()
                elif row[1].partCount > 1:
                    multipartripList.append(row[0])
                    row[2] = row[2][:1]+"A"+row[2][2:]
                    cursor2.updateRow(row)
        #arcpy.AddMessage(multipartList)
        #mp_lyr[0].setSelectionSet(multipartList,'NEW')
        #arcpy.analysis.Select(mp_lyr[0], mpSelect)
        if len(multipartripList) > 0:
            if len(multipartripList) == 1:
                qry = """OBJECTID IN {0})""".format(str(tuple(multipartripList))[:-2])
            else:
                qry = """OBJECTID IN {0}""".format(str(tuple(multipartripList)))
            #arcpy.AddMessage(qry)
            arcpy.conversion.ExportFeatures(mpr_lyr,mprSelect,qry)
        
        return


class overlappingRiparian(object):
    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "Overlapping Riparian"
        self.description = "This tool identifies overlapping riparian polygons, and changes character 6 of QAQC_Code = 'O'. \n Polygonal wetland features in the National Wetlands Inventory are not allowed to overlap, should be reviewed, and correct. The tool exports a new feature class named 'Overlapping Wetlands' to the user's geodatabase to help identify the problematic polygons."
         

        self.canRunInBackground = False
        self.category = "Individual Riparian Tools"

    
    def getParameterInfo(self):
        """Define parameter definitions"""
        # params [0]
        input_gdb = arcpy.Parameter(name = "input_gdb",
                                   displayName = "Input Geodatabase",
                                   datatype = "DEWorkspace",
                                   parameterType = "Required", #Required|Optional|Derived
                                   direction = "Input", #Input|Output
                                   )

        work_area = arcpy.Parameter(name = "work_area",
                                     displayName = "Work Area",
                                     datatype = "String",
                                     parameterType = "Required", #Required|Optional|Derived
                                     direction = "Input", #Input|Output
                                     )
        work_area.value = "CONUS"
        work_area.filter.type = "ValueList"
        work_area.filter.list = ["CONUS","AK","HI","PRVI","PacTrust"]

        #params = None
        
        return [input_gdb, work_area]

    def isLicensed(self):
        """Set whether tool is licensed to execute."""
        return True

    def updateParameters(self, parameters):
        """Modify the values and properties of parameters before internal
        validation is performed.  This method is called whenever a parameter
        has been changed."""

        return

    def updateMessages(self, parameters):
        """Modify the messages created by internal validation for each tool
        parameter.  This method is called after internal validation."""
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        wet_line = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_line".format(work_area=work_area))
        wet_projects = os.path.join(input_gdb,"{work_area}_projects".format(work_area = work_area),"{work_area}_wet_projects".format(work_area=work_area))
        rip_poly = os.path.join(input_gdb,"{work_area}_riparian".format(work_area = work_area),"{work_area}_rip_poly".format(work_area=work_area))
        if not arcpy.Exists(wet_poly):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_poly.".format(work_area=work_area))
        if not arcpy.Exists(wet_projects):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_projects.".format(work_area=work_area))
        if not arcpy.Exists(rip_poly):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_rip_poly.".format(work_area=work_area))

        return

    def execute(self, parameters, messages):
        """The source code of the tool."""

        arcpy.AddMessage("Riparian Overlaps Version 1.4.3")
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)

        arcpy.management.Compact(input_gdb)

        arcpy.AddMessage("Begin Riparian Overlaps tool")
        overlappingRiparian.overlappingWetlands(input_gdb,work_area)

        return

    @staticmethod
    def overlappingWetlands(arg1,arg2):
        input_gdb = str(arg1)
        work_area = str(arg2)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        rip_poly = os.path.join(input_gdb,"{work_area}_riparian".format(work_area = work_area),"{work_area}_rip_poly".format(work_area=work_area))
        #wet_poly_topology = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wetlands_Topology".format(work_area=work_area))
        wet_projects = os.path.join(input_gdb,"{work_area}_projects".format(work_area = work_area),"{work_area}_wet_projects".format(work_area=work_area))
        qaqc_code = "'NNNNN'"
        #arcpy.management.Compact(input_gdb)

        #arcpy.AddMessage("{},{},{},{}".format(input_gdb,wet_poly,wet_poly_topology,wet_projects))

        #rip_intersect = r"memory\rip_intersect"
        rip_intersect = os.path.join(input_gdb,"rip_intersect")
        #wet_intersect = r"memory\wet_intersect"
        wet_intersect = os.path.join(input_gdb,"wet_intersect")
        #combined_intersect = r"memory\combined_intersect"
        combined_intersect = os.path.join(input_gdb,"combined_intersect")

        arcpy.AddMessage("trying to intersect riparian with itself")
        arcpy.AddMessage("rip_intersect: {x1}".format(x1=rip_intersect))
        arcpy.AddMessage("wet_intersect: {x1}".format(x1=wet_intersect))
        arcpy.analysis.Intersect(rip_poly,rip_intersect)
        arcpy.AddMessage("trying to intersect riparian with wet_poly")
        arcpy.analysis.Intersect([rip_poly,wet_poly],wet_intersect)
        arcpy.AddMessage("trying to merge both intersects")
        arcpy.management.Merge([rip_intersect,wet_intersect],combined_intersect)

        arcpy.management.MakeFeatureLayer(rip_poly,"rip_poly_lyr")
        arcpy.management.SelectLayerByLocation("rip_poly_lyr",'INTERSECT',combined_intersect)
        arcpy.conversion.ExportFeatures("rip_poly_lyr",os.path.join(input_gdb,"Overlapping_Riparian"))

        #arcpy.AddMessage("begin cursor")
        fields = ['QAQC_CODE']

        #with arcpy.da.Editor(input_gdb) as edit:
        edit = arcpy.da.Editor(input_gdb)
        edit.startEditing()
        edit.startOperation()
        with arcpy.da.UpdateCursor("rip_poly_lyr",fields) as cursor:
            #arcpy.AddMessage("update cursor processing...")
            for row in cursor:

                if row[0] is None:
                    row[0] = "NNNNN"

                row[0] = row[0][:4]+"O"
                newCode = row[0][:4]+"O"
                #arcpy.AddMessage("newCode = {x3}".format(x3=newCode))
                cursor.updateRow(row)
                #arcpy.AddMessage("updated row")
        edit.stopOperation()
        edit.stopEditing(True)

        arcpy.management.Delete(rip_intersect)
        arcpy.management.Delete(wet_intersect)
        arcpy.management.Delete(combined_intersect)
                                        
        return


class sliverUplands_Riparian(object):
    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "Sliver Uplands Riparian"
        self.description = "Identifies upland islands or holes in wetlands that are less than 0.01 acres. These may be actual upland features but are identified as errors as they are typically errors in wetland mapping. The model changes the fourth character of QAQC_Code = 'U', in all wetland polygons adjacent to the upland sliver. The sliver upland polygons are stored as a new feature class ‘Sliver Uplands’ in your wetlands file geodatabase to assist in locating these small geographic features for review. This tool requires that a 'CONUS_wet_projects' has a feature(s) that defines the wetland mapping project and completely covers all features in the 'CONUS_wet_poly' feature class. NOTE: This tool is a computationally intensive process and may fail on extremely large geographic areas. "
         

        self.canRunInBackground = False
        self.category = "Individual Riparian Tools"

    
    def getParameterInfo(self):
        """Define parameter definitions"""
        # params [0]
        input_gdb = arcpy.Parameter(name = "input_gdb",
                                   displayName = "Input Geodatabase",
                                   datatype = "DEWorkspace",
                                   parameterType = "Required", #Required|Optional|Derived
                                   direction = "Input", #Input|Output
                                   )

        work_area = arcpy.Parameter(name = "work_area",
                                     displayName = "Work Area",
                                     datatype = "String",
                                     parameterType = "Required", #Required|Optional|Derived
                                     direction = "Input", #Input|Output
                                     )
        work_area.filter.type = "ValueList"
        work_area.filter.list = ["CONUS","AK","HI","PRVI","PacTrust"]

        #params = None
        
        return [input_gdb, work_area]
    
    def isLicensed(self):
        """Set whether tool is licensed to execute."""
        return True

    def updateParameters(self, parameters):
        """Modify the values and properties of parameters before internal
        validation is performed.  This method is called whenever a parameter
        has been changed."""

        return

    def updateMessages(self, parameters):
        """Modify the messages created by internal validation for each tool
        parameter.  This method is called after internal validation."""
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        wet_line = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_line".format(work_area=work_area))
        wet_projects = os.path.join(input_gdb,"{work_area}_projects".format(work_area = work_area),"{work_area}_wet_projects".format(work_area=work_area))
        rip_poly = os.path.join(input_gdb,"{work_area}_riparian".format(work_area = work_area),"{work_area}_rip_poly".format(work_area=work_area))
        if not arcpy.Exists(wet_poly):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_poly.".format(work_area=work_area))
        if not arcpy.Exists(wet_projects):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_projects.".format(work_area=work_area))
        if not arcpy.Exists(rip_poly):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_rip_poly.".format(work_area=work_area))

        return

    def execute(self, parameters, messages):
        """The source code of the tool."""

        arcpy.AddMessage("Riparian Sliver Uplands Version 1.4.3")
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        #wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        #wet_projects = os.path.join(input_gdb,"{work_area}_projects".format(work_area = work_area),"{work_area}_wet_projects".format(work_area=work_area))

        arcpy.management.Compact(input_gdb)

        arcpy.AddMessage("Begin Riparian Sliver Uplands tool")
        sliverUplands_Riparian.sliverUplands(input_gdb,work_area)
        
        return

    @staticmethod
    def sliverUplands(arg1,arg2):
        arcpy.env.maintainCurveSegments = True
        input_gdb = str(arg1)
        work_area = str(arg2)
        
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        rip_poly = os.path.join(input_gdb,"{work_area}_riparian".format(work_area = work_area),"{work_area}_rip_poly".format(work_area=work_area))
        wet_projects = os.path.join(input_gdb,"{work_area}_projects".format(work_area = work_area),"{work_area}_wet_projects".format(work_area=work_area))

        qaqc_code = "'NNNNN'"
        #arcpy.management.Compact(input_gdb)

        arcpy.management.MakeFeatureLayer(wet_poly,"wet_poly_lyr")
        arcpy.management.MakeFeatureLayer(rip_poly,"rip_poly_lyr")
        arcpy.management.SelectLayerByLocation("wet_poly_lyr","INTERSECT","rip_poly_lyr")
        arcpy.CalculateField_management(rip_poly,"ACRES","!shape.area@acres!","PYTHON","")
        
        #Try the cursor method here
        fields = ['QAQC_CODE']
        #arcpy.AddMessage("begin cursor")
        edit = arcpy.da.Editor(input_gdb)
        edit.startEditing()
        edit.startOperation()
        with arcpy.da.UpdateCursor(rip_poly,fields) as cursor:
            #arcpy.AddMessage("update cursor processing...")
            for row in cursor:
                row[0] = row[0][:3]+"N"+row[0][4:]
                cursor.updateRow(row)
                #arcpy.AddMessage("updated row")

        edit.stopOperation()
        edit.stopEditing(True)

        wet_rip_merge = r"memory\wet_rip_merge"

        arcpy.management.Merge(["wet_poly_lyr","rip_poly_lyr"],wet_rip_merge)
        
        tempDissolve = r"memory\tempDissolve"

        arcpy.management.Dissolve(wet_rip_merge,tempDissolve,"","","SINGLE_PART")
        #arcpy.analysis.PairwiseDissolve("wet_poly_lyr",tempDissolve,"","","SINGLE_PART")

        arcpy.management.AddField(tempDissolve,"PART_COUNT","DOUBLE")

        fields = ['PART_COUNT','SHAPE@']
        #arcpy.AddMessage("begin tempDissolve cursor")
        edit = arcpy.da.Editor(input_gdb)
        edit.startEditing()
        edit.startOperation()
        with arcpy.da.UpdateCursor(tempDissolve,fields) as cursor:
            #arcpy.AddMessage("update cursor processing...")
            for row in cursor:
                if row[1] is None:
                    cursor.deleteRow()
                else:
                    parts = row[1].partCount
                    boundaries = row[1].boundary().partCount
                    if boundaries > parts:
                        row[0] = 1
                    else:
                        row[0] = 0
                    cursor.updateRow(row)
                #arcpy.AddMessage("updated row")

        edit.stopOperation()
        edit.stopEditing(True)

        arcpy.management.MakeFeatureLayer(tempDissolve,"tempDissolve_lyr")

        arcpy.management.SelectLayerByAttribute("tempDissolve_lyr","NEW_SELECTION",'"PART_COUNT" > 0')

        #dissolveDonuts = os.path.join(input_gdb,"dissolveDonuts")
        dissolveDonuts = r"memory\dissolveDonuts"
        arcpy.management.CopyFeatures("tempDissolve_lyr",dissolveDonuts)

        #dissolveDonuts_union = os.path.join(input_gdb,"dissolveDonuts_union")
        dissolveDonuts_union = r"memory\dissolveDonuts_union"
        arcpy.analysis.Union(dissolveDonuts,dissolveDonuts_union,"","","NO_GAPS")

        arcpy.management.AddField(dissolveDonuts_union,"ACRES","DOUBLE")
        arcpy.CalculateField_management(dissolveDonuts_union,"ACRES","!shape.area@acres!","PYTHON","")

        arcpy.management.MakeFeatureLayer(dissolveDonuts_union,"Sliver_rip_uplands_lyr")
        arcpy.management.SelectLayerByAttribute("Sliver_rip_uplands_lyr","NEW_SELECTION",'"FID_dissolveDonuts" = -1 AND "ACRES" <= 0.01')
        arcpy.management.DeleteField("Sliver_rip_uplands_lyr",["FID_dissolveDonuts"])
        arcpy.management.CopyFeatures("Sliver_rip_uplands_lyr",os.path.join(input_gdb,"Sliver_Uplands_Riparian"))

        arcpy.management.SelectLayerByLocation("rip_poly_lyr","INTERSECT","Sliver_rip_uplands_lyr")

        fields = ['QAQC_CODE']
        #arcpy.AddMessage("begin QAQC Code cursor")
        edit = arcpy.da.Editor(input_gdb)
        edit.startEditing()
        edit.startOperation()
        with arcpy.da.UpdateCursor("rip_poly_lyr",fields) as cursor:
            #arcpy.AddMessage("update cursor processing...")
            for row in cursor:
                row[0] = row[0][:3]+"U"+row[0][4:]

                cursor.updateRow(row)
                #arcpy.AddMessage("updated row")

        edit.stopOperation()
        edit.stopEditing(True)


        wet_projects_line = r"memory\projectLine"
        wet_projects_line_5m = os.path.join(input_gdb,"projects_line_5m")
        wet_projects_line_point01m = os.path.join(input_gdb,"projects_line_point01m")
        rip_poly_clipped = os.path.join(input_gdb,"rip_poly_clipped")

        arcpy.management.PolygonToLine(wet_projects,wet_projects_line)
        arcpy.analysis.Buffer(wet_projects_line,wet_projects_line_5m,"5 Meters")
        arcpy.analysis.Buffer(wet_projects_line,wet_projects_line_point01m,".01 Meters")
        arcpy.analysis.Clip(rip_poly,wet_projects_line_5m,rip_poly_clipped)
        
        arcpy.management.MakeFeatureLayer(rip_poly_clipped,"rip_poly_clipped_lyr")
        arcpy.management.SelectLayerByLocation("rip_poly_clipped_lyr","WITHIN_A_DISTANCE",wet_projects_line, '5 Meters')
        arcpy.management.SelectLayerByLocation("rip_poly_clipped_lyr","INTERSECT",wet_projects_line,"","REMOVE_FROM_SELECTION")
        arcpy.management.SelectLayerByLocation("rip_poly_lyr","CONTAINS","rip_poly_clipped_lyr")
        #project_erase = os.path.join(input_gdb,"projectErase")
        #arcpy.analysis.PairwiseErase(wet_projects,"wet_poly_lyr",project_erase)
        arcpy.management.CopyFeatures("rip_poly_lyr",os.path.join(input_gdb,"Potential_Riparian_Upland_Gaps"))
        #arcpy.analysis.Buffer("wet_poly_lyr",os.path.join(input_gdb,"poly_buffered"),".01 Meters")
        #arcpy.analysis.Intersect([wet_projects_line_point01m,os.path.join(input_gdb,"poly_buffered")],os.path.join(input_gdb,"gap_intersect"))

        fields = ['QAQC_CODE']
        #arcpy.AddMessage("begin QAQC Code cursor")
        edit = arcpy.da.Editor(input_gdb)
        edit.startEditing()
        edit.startOperation()
        with arcpy.da.UpdateCursor("rip_poly_lyr",fields) as cursor:
            #arcpy.AddMessage("update cursor processing...")
            for row in cursor:
                row[0] = row[0][:3]+"G"+row[0][4:]

                cursor.updateRow(row)
                #arcpy.AddMessage("updated row")

        edit.stopOperation()
        edit.stopEditing(True)

        if arcpy.Exists(wet_projects_line_5m):
            arcpy.management.Delete(wet_projects_line_5m)
        if arcpy.Exists(wet_projects_line_point01m):
            arcpy.management.Delete(wet_projects_line_point01m)
        if arcpy.Exists(rip_poly_clipped):
            arcpy.management.Delete(rip_poly_clipped)
        
        return

class QAQCSummary_Riparian(object):
    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "QAQC Summary - Riparian"
        self.description = "Summarizes the QAQC_CODE field into a 'QAQC_Summary' table in your wetlands file geodatabase. It also describes each error type and records who conducted the verification and when the verification was run. Comments can be added to the ‘comments’ field of the QAQC_Summary table to justify specific types of errors. These comments will only be saved if the ‘Append to History Table’ box is checked prior to running the All_QAQC_Checks model. "
         

        self.canRunInBackground = False
        self.category = "Individual Riparian Tools"

    
    def getParameterInfo(self):
        """Define parameter definitions"""
        # params [0]
        input_gdb = arcpy.Parameter(name = "input_gdb",
                                   displayName = "Input Geodatabase",
                                   datatype = "DEWorkspace",
                                   parameterType = "Required", #Required|Optional|Derived
                                   direction = "Input", #Input|Output
                                   )

        work_area = arcpy.Parameter(name = "work_area",
                                     displayName = "Work Area",
                                     datatype = "String",
                                     parameterType = "Required", #Required|Optional|Derived
                                     direction = "Input", #Input|Output
                                     )
        work_area.value = "CONUS"
        work_area.filter.type = "ValueList"
        work_area.filter.list = ["CONUS","AK","HI","PRVI","PacTrust"]

        verifier_name = arcpy.Parameter(name = "verifier_name",
                                        displayName = "Verified By:",
                                        datatype = "String",
                                        parameterType = "Required",
                                        direction = "Input",
                                        )

        save_summary_table = arcpy.Parameter(name = "save_summary_table",
                                             displayName = "Save Summary Table?",
                                             datatype = "GPBoolean",
                                             parameterType = "Required",
                                             direction = "Input",
                                             )
        save_summary_table.value = True
        #params = None
        
        return [input_gdb, work_area,verifier_name,save_summary_table]

    def isLicensed(self):
        """Set whether tool is licensed to execute."""
        return True

    def updateParameters(self, parameters):
        """Modify the values and properties of parameters before internal
        validation is performed.  This method is called whenever a parameter
        has been changed."""

        return

    def updateMessages(self, parameters):
        """Modify the messages created by internal validation for each tool
        parameter.  This method is called after internal validation."""
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        wet_line = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_line".format(work_area=work_area))
        wet_projects = os.path.join(input_gdb,"{work_area}_projects".format(work_area = work_area),"{work_area}_wet_projects".format(work_area=work_area))
        rip_poly = os.path.join(input_gdb,"{work_area}_riparian".format(work_area = work_area),"{work_area}_rip_poly".format(work_area=work_area))
        if not arcpy.Exists(rip_poly):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_rip_poly.".format(work_area=work_area))

        return

    def execute(self, parameters, messages):
        """The source code of the tool."""

        arcpy.AddMessage("Riparian QAQC Summary Version 1.4.3")
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        verifier_name = str(parameters[2].value)
        save_summary_table = str(parameters[3].value)
        #wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        
        arcpy.management.Compact(input_gdb)
        
        arcpy.AddMessage("Begin Riparian QAQC Summary tool")
        QAQCSummary_Riparian.QAQCSummary_tool(input_gdb,work_area,verifier_name,save_summary_table)
                        
        return

    @staticmethod
    def QAQCSummary_tool(arg1,arg2,arg3,arg4):

        input_gdb = str(arg1)
        work_area = str(arg2)
        verifier_name = str(arg3)
        save_summary_table = arg4
        #arcpy.AddMessage("arg4: {x1},type(arg4): {x2}".format(x1=arg4,x2=type(arg4)))
        rip_poly = os.path.join(input_gdb,"{work_area}_riparian".format(work_area = work_area),"{work_area}_rip_poly".format(work_area=work_area))
        summaryTable = os.path.join(input_gdb,"QAQC_Riparian_Summary")
        #arcpy.management.Compact(input_gdb)
        
        arcpy.analysis.Frequency(rip_poly,summaryTable,["QAQC_CODE"])
        arcpy.management.AddField(summaryTable,"ERROR_TYPE","TEXT","","",255)
        arcpy.management.AddField(summaryTable,"VERIFIED_BY","TEXT","","",255)
        arcpy.management.AddField(summaryTable,"VERIFICATION_DATE","DATE")
        arcpy.management.AddField(summaryTable,"VERIFICATION_COMMENTS","TEXT","","",255)
        
        
        #arcpy.AddMessage("begin cursor")
        fields = ['QAQC_CODE','ERROR_TYPE']
        

        edit = arcpy.da.Editor(input_gdb)
        edit.startEditing()
        edit.startOperation()
        with arcpy.da.UpdateCursor(summaryTable,fields) as cursor:
            #arcpy.AddMessage("update cursor processing...")
            for row in cursor:
                newList = []

                if row[0] == "NNNNN":
                    row[1] = "Passed Verification"
                else:
                    printDict = {
                        'C': 'Bad Code',
                        'A': 'Adjacent Polygon',
                        'S': 'Sliver Riparian',
                        'U': 'Sliver Upland',
                        'G': 'Gap Riparian',
                        'O': 'Overlapping Riparian',
                        ',': ', '
                        }
                    #newList = []
                    #newListString = ''
                    for i,v in enumerate(row[0]):
                        #arcpy.AddMessage(v)
                        #arcpy.AddMessage(type(newList))
                        if v != 'N':
                            newList.append(v)
                            
                        newListComma = ','.join(newList)
                        #arcpy.AddMessage('newListComma: {x}'.format(x=newListComma))
                        
                        printString = ''
                        for char in newListComma:
                            nLC = str(printDict[char])
                            printString += nLC

                        #arcpy.AddMessage('printString: {y}'.format(y=printString))
                        row[1] = printString
                    
                cursor.updateRow(row)
                #arcpy.AddMessage("updated row")
                
        edit.stopOperation()
        edit.stopEditing(True)

        arcpy.CalculateField_management(summaryTable,"VERIFIED_BY","'"+verifier_name+"'","PYTHON","")
        #date_value = datetime.datetime.now()
        arcpy.CalculateField_management(summaryTable,"VERIFICATION_DATE","datetime.datetime.now()","PYTHON","")

        history_table = os.path.join(input_gdb,"QAQC_Riparian_History")
        if arcpy.Exists(history_table):
            run_list = []
            with arcpy.da.SearchCursor(history_table,'RUN') as cursor:
                for row in cursor:
                    run_list.append(row[0])
            run_max = max(run_list)
            #arcpy.AddMessage("run_max: {x1}".format(x1=run_max))
        
            #arcpy.AddMessage("history table exists")

            arcpy.AddField_management(summaryTable,"RUN","SHORT")
            arcpy.CalculateField_management(summaryTable,"RUN",run_max+1)
            arcpy.management.Append(summaryTable,history_table,"TEST")
        else:
            arcpy.AddField_management(summaryTable,"RUN","SHORT")
            arcpy.CalculateField_management(summaryTable,"RUN",1)
            #arcpy.CopyFeatures_management(summaryTable,history_table)
            arcpy.conversion.TableToTable(summaryTable,input_gdb,'QAQC_Riparian_History')


        return



class incorrectRiparianCodes(object):
    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "Incorrect Riparian Codes"
        self.description = "Identifies riparian polygons with incorrect riparian codes, or null or blank values in the 'attribute' field. A bad attribute summary table is created and stored with your wetlands file geodatabase. Changes character 1 of QAQC_Code = 'C' if wetland code is bad. "
         

        self.canRunInBackground = False
        self.category = "Individual Riparian Tools"

    
    def getParameterInfo(self):
        """Define parameter definitions"""
        # params [0]
        input_gdb = arcpy.Parameter(name = "input_gdb",
                                   displayName = "Input Geodatabase",
                                   datatype = "DEWorkspace",
                                   parameterType = "Required", #Required|Optional|Derived
                                   direction = "Input", #Input|Output
                                   )

        work_area = arcpy.Parameter(name = "work_area",
                                     displayName = "Work Area",
                                     datatype = "String",
                                     parameterType = "Required", #Required|Optional|Derived
                                     direction = "Input", #Input|Output
                                     )
        work_area.value = "CONUS"
        work_area.filter.type = "ValueList"
        work_area.filter.list = ["CONUS","AK","HI","PRVI","PacTrust"]

        #params = None
        
        return [input_gdb, work_area]

    def isLicensed(self):
        """Set whether tool is licensed to execute."""
        return True

    def updateParameters(self, parameters):
        """Modify the values and properties of parameters before internal
        validation is performed.  This method is called whenever a parameter
        has been changed."""

        return

    def updateMessages(self, parameters):
        """Modify the messages created by internal validation for each tool
        parameter.  This method is called after internal validation."""
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        wet_line = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_line".format(work_area=work_area))
        wet_projects = os.path.join(input_gdb,"{work_area}_projects".format(work_area = work_area),"{work_area}_wet_projects".format(work_area=work_area))
        rip_poly = os.path.join(input_gdb,"{work_area}_riparian".format(work_area = work_area),"{work_area}_rip_poly".format(work_area=work_area))
        if not arcpy.Exists(rip_poly):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_rip_poly.".format(work_area=work_area))

        return

    def execute(self, parameters, messages):
        """The source code of the tool."""

        arcpy.AddMessage("Incorrect Riparian Codes Version 1.4.3")
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        #wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        
        arcpy.management.Compact(input_gdb)
        
        arcpy.AddMessage("Begin Incorrect Riparian Codes tool")
        incorrectRiparianCodes.incorrectCodes(input_gdb,work_area)
                        
        return

    @staticmethod
    def incorrectCodes(arg1,arg2):
        input_gdb = str(arg1)
        work_area = str(arg2)
        rip_poly = os.path.join(input_gdb,"{work_area}_riparian".format(work_area = work_area),"{work_area}_rip_poly".format(work_area=work_area))

        arcpy.management.MakeFeatureLayer(rip_poly,"rip_poly_lyr")

        fields = ['QAQC_CODE','ATTRIBUTE']

        edit = arcpy.da.Editor(input_gdb)
        edit.startEditing()
        edit.startOperation()
        with arcpy.da.UpdateCursor("rip_poly_lyr",fields) as cursor:
            #arcpy.AddMessage("update cursor processing...")
            for row in cursor:

                if row[0] is None:
                    row[0] = "NNNNNN"
                
                row[0] = "N"+row[0][1:]
                newCode = "N"+row[0][1:]
                #arcpy.AddMessage("newCode = {x3}".format(x3=newCode))
                
                #arcpy.AddMessage("updated row")

                if row[1] is not None:
                    row[1] = row[1].strip(" ")
                cursor.updateRow(row)
                
        edit.stopOperation()
        edit.stopEditing(True)

        goodCodeList = []
        badCodeList = []

        #list_fields = ['ATTRIBUTE','QAQC_CODE']
        list_fields = ['ATTRIBUTE']
        #arcpy.AddMessage("begin ATTRIBUTE cursor")

        unique_codes = r"memory\ripUniqueCodes"

        arcpy.analysis.Frequency(rip_poly, unique_codes, "ATTRIBUTE", None)

        with arcpy.da.SearchCursor(unique_codes,list_fields) as cursor:
            for row in cursor:
                code = str(row[0])
                #arcpy.AddMessage(code)
                incorrectRiparianCodes.codeCheck(code,goodCodeList,badCodeList)


        if len(badCodeList) > 0:
            arcpy.management.MakeFeatureLayer(rip_poly,"rip_poly_lyr")
            #qry = """ATTRIBUTE IN {0}""".format(str(tuple(badCodeList)))
            if len(badCodeList) == 1:
                qry = """ATTRIBUTE IN {0})""".format(str(tuple(badCodeList))[:-2])
            else:
                qry = """ATTRIBUTE IN {0}""".format(str(tuple(badCodeList)))
            arcpy.AddMessage(qry)
            arcpy.AddMessage("bad codes: {x1}".format(x1=badCodeList))
        
            arcpy.management.SelectLayerByAttribute("rip_poly_lyr","NEW_SELECTION",qry)
            arcpy.management.SelectLayerByAttribute("rip_poly_lyr","ADD_TO_SELECTION","ATTRIBUTE IS NULL")

            fields = ['QAQC_CODE']
            edit = arcpy.da.Editor(input_gdb)
            edit.startEditing()
            edit.startOperation()
            with arcpy.da.UpdateCursor("rip_poly_lyr",fields) as cursor:
                #arcpy.AddMessage("update cursor processing...")
                for row in cursor:
                    if row[0] is None:
                        row[0] = "NNNNN"

                    row[0] = "C"+row[0][1:]

                    cursor.updateRow(row)
                    #arcpy.AddMessage("updated row")

            edit.stopOperation()
            edit.stopEditing(True)
            
            arcpy.AddMessage("Incorrect Riparian Codes: {bCL}".format(bCL=badCodeList))

        return

    @staticmethod
    def codeCheck(code,goodCodeList,badCodeList):

        '''
        Early Checks
        Check if the code has already been determined to be good, cut down on processing time
        Check if code is long enough
        Check if code has invalid characters
        '''
        split_check = False
        #arcpy.AddMessage("code: {x1}".format(x1=code))

        if code in goodCodeList:
            #print("good, already checked this one")
            return 'good'
        if code in badCodeList:
            #print("no good, you already checked this one")
            badCodeList.append(code)
            return code

        if code is None:
            badCodeList.append(code)
            return code

        if len(code)<5:
            #arcpy.AddMessage("no good - not enough characters")
            badCodeList.append(code)
            return code

        if len(code)<6:
            if code not in ('Rp1EM','Rp1SS','Rp1FO','Rp2EM','Rp2SS','Rp2FO'):
                badCodeList.append(code)
                return code
            else:
                goodCodeList.append(code)
                return code
        
        if r'/' in code:
            split_check = True
            #print("buddy, i don't know we're working on it")
            split_position = code.find(r'/')
            #print(split_position)
            if code.count(r'/') > 1:
                #arcpy.AddMessage("hey now, too many splits")
                badCodeList.append(code)
                return code
            if len(code)<split_position+3:
                #arcpy.AddMessage("need more characters after the split")
                badCodeList.append(code)
                return code
            
        if not code.isalnum() and r'/' not in code:
            #arcpy.AddMessage("no good - invalid character(s)")
            badCodeList.append(code)
            return code

        #pd = codeParse(code)
        '''
        System/Subsystem Checks
        '''

        systemList = ['Rp']
        subSystemList = ['Rp1','Rp2']

        #system = pd['system']
        #print(system)

        if code[:3] not in subSystemList:
            #arcpy.AddMessage("no good - system or subsystem")
            badCodeList.append(code)
            return code

        '''
        Class Checks
        '''
        
        classDict = {
            'Rp1': ['FO','SS','EM'],
            'Rp2': ['FO','SS','EM'],
            }

        if code[3:5] not in classDict[code[:3]]:
            #print("no good - class")
            badCodeList.append(code)
            return code
        
        '''
        Subclass Checks
        '''
        
        subclassDict = {
            'SS': ['5','6','7','8'],
            'FO': ['5','6','7','8']
            }

        if code[3:5] in ('SS','FO'):
            #arcpy.AddMessage("no good - subclass")
            if code[5].isnumeric() and code[5] not in subclassDict[code[3:5]]:
                arcpy.AddMessage('a')
                arcpy.AddMessage("no good - subclass")
                badCodeList.append(code)
                return code

        '''
        Dominance Type Checks
        '''

        #dictKeys are characters [3:5]
        #dictValues are characters [6:8] or [5:7] for EM
        dominanceTypeDict = {
            'FO': [],
            'FO5': ['SY','CW','SC','MQ','AS','AL','RO','WI','MD','JU','WS','EO','BS','ME'],
            'FO6': ['SY','CW','SC','MQ','AS','AL','RO','WI','MD'],
            'FO7': ['JU','WS','EO','BS','ME'],
            'FO8': [],
            'SS': [],
            'SS5': ['SY','CW','SC','MQ','AS','AL','RO','WI','BB','GW','RB','MD','JU','WS','EO','BS','SB','ME'],
            'SS6': ['SY','CW','SC','MQ','AS','AL','RO','WI','BB','GW','RB','MD'],
            'SS7': ['JU','WS','EO','BS','SB','ME'],
            'SS8': [],
            'EM': ['AK','WW','GB']
            }
            
        
        if not split_check:

            if code[5].isnumeric():
                if len(code)==7:
                    #arcpy.AddMessage('b')
                    badCodeList.append(code)
                    return code
                if len(code)==8 and code[6:8] not in dominanceTypeDict[code[3:6]]:
                    arcpy.AddMessage("no good - dominance type")
                    arcpy.AddMessage('c')
                    arcpy.AddMessage(code)
                    arcpy.AddMessage(len(code))
                    arcpy.AddMessage("code[6:8]: {x1}".format(x1=code[6:8]))
                    arcpy.AddMessage("dominanceTypeDict[code[3:5]]: {x2}".format(x2=dominanceTypeDict[code[3:6]]))
                    badCodeList.append(code)
                    return code
                if len(code)>8:
                    #arcpy.AddMessage('d')
                    badCodeList.append(code)
                    return code
            elif code[5].isalpha():
                if len(code)==6:
                    #arcpy.AddMessage('e')
                    badCodeList.append(code)
                    return code
                if len(code)==7 and code[5:7] not in dominanceTypeDict[code[3:6]]:
                    #arcpy.AddMessage('f')
                    badCodeList.append(code)
                    return code
                if len(code)>7:
                    #arcpy.AddMessage('g')
                    badCodeList.append(code)
                    return code
                
        else:
            pre_split_code = code[:split_position]
            post_split_code = code[:3]+code[split_position+1:]

            if len(pre_split_code)<6:
                if code not in ('Rp1EM','Rp1SS','Rp1FO','Rp2EM','Rp2SS','Rp2FO'):
                    badCodeList.append(code)
                    return code

            if len(post_split_code)<6:
                if code not in ('Rp1EM','Rp1SS','Rp1FO','Rp2EM','Rp2SS','Rp2FO'):
                    badCodeList.append(code)
                    return code
            
            if pre_split_code[5].isnumeric():
                if len(pre_split_code)==7:
                    #arcpy.AddMessage('h')
                    badCodeList.append(code)
                    return code
                if len(pre_split_code)==8 and pre_split_code[6:8] not in dominanceTypeDict[pre_split_code[3:6]]:
                    #print("no good - dominance type")
                    #arcpy.AddMessage('i')
                    badCodeList.append(code)
                    return code
                if len(pre_split_code)>8:
                    #arcpy.AddMessage('j')
                    badCodeList.append(code)
                    return code
            elif pre_split_code[5].isalpha():
                if len(pre_split_code)==6:
                    #arcpy.AddMessage('k')
                    badCodeList.append(code)
                    return code
                if len(pre_split_code)==7 and pre_split_code[5:7] not in dominanceTypeDict[pre_split_code[3:6]]:
                    #arcpy.AddMessage('l')
                    badCodeList.append(code)
                    return code
                if len(pre_split_code)>7:
                    #arcpy.AddMessage('m')
                    badCodeList.append(code)
                    return code

            if post_split_code[5].isnumeric():
                if len(post_split_code)==7:
                    #arcpy.AddMessage('n')
                    badCodeList.append(code)
                    return code
                if len(post_split_code)==8 and post_split_code[6:8] not in dominanceTypeDict[post_split_code[3:6]]:
                    #print("no good - dominance type")
                    #arcpy.AddMessage('o')
                    badCodeList.append(code)
                    return code
                if len(post_split_code)>8:
                    #arcpy.AddMessage('p')
                    badCodeList.append(code)
                    return code
            elif post_split_code[5].isalpha():
                if len(post_split_code)==6:
                    #arcpy.AddMessage('q')
                    badCodeList.append(code)
                    return code
                if len(post_split_code)==7 and post_split_code[5:7] not in dominanceTypeDict[post_split_code[3:6]]:
                    #arcpy.AddMessage('r')
                    badCodeList.append(code)
                    return code
                if len(post_split_code)>7:
                    #arcpy.AddMessage('s')
                    badCodeList.append(code)
                    return code

        goodCodeList.append(code)

        return 'good'

class combinedRiparianTools(object):
    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "Combined Riparian Tools"
        self.description = "This tool runs all the riparian QAQC Tools."
        self.canRunInBackground = False
        self.category = "Combined Tools"
            
    def getParameterInfo(self):
        """Define parameter definitions"""
        # params [0]
        input_gdb = arcpy.Parameter(name = "input_gdb",
                                   displayName = "Input Geodatabase",
                                   datatype = "DEWorkspace",
                                   parameterType = "Required", #Required|Optional|Derived
                                   direction = "Input", #Input|Output
                                   )

        work_area = arcpy.Parameter(name = "work_area",
                                     displayName = "Work Area",
                                     datatype = "String",
                                     parameterType = "Required", #Required|Optional|Derived
                                     direction = "Input", #Input|Output
                                     )

        work_area.value = "CONUS"
        work_area.filter.type = "ValueList"
        work_area.filter.list = ["CONUS","AK","HI","PRVI","PacTrust"]

        verifier_name = arcpy.Parameter(name = "verifier_name",
                                        displayName = "Verified By:",
                                        datatype = "String",
                                        parameterType = "Required",
                                        direction = "Input",
                                        )

        save_summary_table = arcpy.Parameter(name = "save_summary_table",
                                             displayName = "Save Summary Table?",
                                             datatype = "GPBoolean",
                                             parameterType = "Required",
                                             direction = "Input",
                                             )
        save_summary_table.value = True

        #params = None
        return [input_gdb,work_area,verifier_name,save_summary_table]

    def isLicensed(self):
        """Set whether tool is licensed to execute."""
        return True

    def updateParameters(self, parameters):
        """Modify the values and properties of parameters before internal
        validation is performed.  This method is called whenever a parameter
        has been changed."""

        return

    def updateMessages(self, parameters):
        """Modify the messages created by internal validation for each tool
        parameter.  This method is called after internal validation."""
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        wet_line = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_line".format(work_area=work_area))
        wet_projects = os.path.join(input_gdb,"{work_area}_projects".format(work_area = work_area),"{work_area}_wet_projects".format(work_area=work_area))
        rip_poly = os.path.join(input_gdb,"{work_area}_riparian".format(work_area = work_area),"{work_area}_rip_poly".format(work_area=work_area))
        if not arcpy.Exists(wet_poly):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_poly.".format(work_area=work_area))
        if not arcpy.Exists(wet_projects):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_projects.".format(work_area=work_area))
        if not arcpy.Exists(rip_poly):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_rip_poly.".format(work_area=work_area))

        return

    def execute(self, parameters, messages):
        """The source code of the tool."""

        arcpy.AddMessage("Combined Riparian Tools Version 1.4.3")
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        wet_line = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_line".format(work_area=work_area))
        rip_poly = os.path.join(input_gdb,"{work_area}_riparian".format(work_area = work_area),"{work_area}_rip_poly".format(work_area=work_area))
        wet_projects = os.path.join(input_gdb,"{work_area}_projects".format(work_area = work_area),"{work_area}_wet_projects".format(work_area=work_area))

        arcpy.management.Compact(input_gdb)

        verifier_name = str(parameters[2].value)
        save_summary_table = str(parameters[3].value)

        #arcpy.AddMessage(verifier_name)
        #arcpy.AddMessage(save_summary_table)

        arcpy.AddMessage("Resetting Riparian QAQC code and calculating Riparian Type")
        QAQCcodeReset_Riparian.QAQCcodeReset_tool(input_gdb,work_area)
        riparianTypeCalc.riparianTypeCalc(input_gdb,work_area)
        
        create_NWIID.create_NWIID_tool(input_gdb,work_area,"rip")

        arcpy.AddMessage("Begin Incorrect Riparian Codes tool")
        incorrectRiparianCodes.incorrectCodes(input_gdb,work_area)
        arcpy.AddMessage("Begin Adjacent Riparian tool")
        adjacentRiparian.adjacentRiparian(input_gdb,work_area)
        arcpy.AddMessage("Begin Sliver Riparian tool")
        sliverRiparian.sliverRiparian_tool(input_gdb,work_area)
        arcpy.AddMessage("Begin Sliver Uplands - Riparian tool")
        sliverUplands_Riparian.sliverUplands(input_gdb,work_area)
        arcpy.AddMessage("Begin Overlapping Riparian tool")
        overlappingRiparian.overlappingWetlands(input_gdb,work_area)
               
        arcpy.AddMessage("Begin Riparian QAQC Summary tool")
        QAQCSummary_Riparian.QAQCSummary_tool(input_gdb,work_area,verifier_name,save_summary_table)        
        
        return

class combinedWetlandRiparianTools(object):
    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "Combined Wetland and Riparian Tools"
        self.description = "This tool runs all the wetland polygonal and linear and riparian QAQC Tools."
        self.canRunInBackground = False
        self.category = "Combined Tools"
            
    def getParameterInfo(self):
        """Define parameter definitions"""
        # params [0]
        input_gdb = arcpy.Parameter(name = "input_gdb",
                                   displayName = "Input Geodatabase",
                                   datatype = "DEWorkspace",
                                   parameterType = "Required", #Required|Optional|Derived
                                   direction = "Input", #Input|Output
                                   )

        work_area = arcpy.Parameter(name = "work_area",
                                     displayName = "Work Area",
                                     datatype = "String",
                                     parameterType = "Required", #Required|Optional|Derived
                                     direction = "Input", #Input|Output
                                     )

        work_area.value = "CONUS"
        work_area.filter.type = "ValueList"
        work_area.filter.list = ["CONUS","AK","HI","PRVI","PacTrust"]

        verifier_name = arcpy.Parameter(name = "verifier_name",
                                        displayName = "Verified By:",
                                        datatype = "String",
                                        parameterType = "Required",
                                        direction = "Input",
                                        )

        save_summary_table = arcpy.Parameter(name = "save_summary_table",
                                             displayName = "Save Summary Table?",
                                             datatype = "GPBoolean",
                                             parameterType = "Required",
                                             direction = "Input",
                                             )
        intermediate_data = arcpy.Parameter(name = "intermediate_data_type",
                                            displayName = "Temporary Data Type",
                                            datatype = "GPBoolean",
                                            parameterType = "Required",
                                            direction = "Input"
                                            )

        intermediate_data.value = False
        
        project_division = arcpy.Parameter(name = "project_division_number",
                                           displayName = "Number of Processing Divisions",
                                           datatype = "GPLong",
                                           parameterType = "Required",
                                           direction = "Input")

        project_division.value = 1
        project_division.filter.type = "ValueList"
        project_division.filter.list = [1,4,9,16,25]

        #params = None
        return [input_gdb,work_area,verifier_name,save_summary_table,intermediate_data,project_division]

    def isLicensed(self):
        """Set whether tool is licensed to execute."""
        return True

    def updateParameters(self, parameters):
        """Modify the values and properties of parameters before internal
        validation is performed.  This method is called whenever a parameter
        has been changed."""

        return

    def updateMessages(self, parameters):
        """Modify the messages created by internal validation for each tool
        parameter.  This method is called after internal validation."""
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        wet_line = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_line".format(work_area=work_area))
        wet_projects = os.path.join(input_gdb,"{work_area}_projects".format(work_area = work_area),"{work_area}_wet_projects".format(work_area=work_area))
        rip_poly = os.path.join(input_gdb,"{work_area}_riparian".format(work_area = work_area),"{work_area}_rip_poly".format(work_area=work_area))
        if not arcpy.Exists(wet_poly):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_poly.".format(work_area=work_area))
        if not arcpy.Exists(wet_projects):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_wet_projects.".format(work_area=work_area))
        if not arcpy.Exists(rip_poly):
            parameters[0].setErrorMessage("The geodatabase must be in the NWI schema to run this tool. The input geodatabase does not have {work_area}_rip_poly.".format(work_area=work_area))

        return

    def execute(self, parameters, messages):
        """The source code of the tool."""

        arcpy.AddMessage("Combined Wetland and Riparian Tools Version 1.4.3")
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        wet_line = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_line".format(work_area=work_area))
        rip_poly = os.path.join(input_gdb,"{work_area}_riparian".format(work_area = work_area),"{work_area}_rip_poly".format(work_area=work_area))
        wet_projects = os.path.join(input_gdb,"{work_area}_projects".format(work_area = work_area),"{work_area}_wet_projects".format(work_area=work_area))

        arcpy.management.Compact(input_gdb)

        verifier_name = str(parameters[2].value)
        save_summary_table = str(parameters[3].value)
        intermediate_data = parameters[4].value
        project_division = parameters[5].value

        arcpy.AddMessage(verifier_name)
        arcpy.AddMessage(save_summary_table)

        #Polygonal Tools
        arcpy.AddMessage("Beginning Polygonal Tools: ")
        QAQCcodeReset_poly.QAQCcodeReset_tool(input_gdb,work_area)
        wetlandTypeCalc_poly.wetlandTypeCalc(input_gdb,work_area)
        create_NWIID.create_NWIID_tool(input_gdb,work_area,"wet")
        
        
        sliverWetlands_poly.sliverWetlands(input_gdb,work_area)
        overlappingWetlands_poly.overlappingWetlands(input_gdb,work_area)
        lakeAndPondSize.lakeAndPondSize(input_gdb,work_area,intermediate_data)
        adjacentWetlands.adjacentWetlands(input_gdb,work_area,intermediate_data)
        sliverUplands_poly.sliverUplands(input_gdb,work_area,intermediate_data,project_division)
        incorrectWetlandsCodes_poly.incorrectCodes(input_gdb,work_area)

        #Linear Tools
        if arcpy.Exists(wet_line):

            linear_count = arcpy.management.GetCount(wet_line).getOutput(0)
            if int(linear_count) > 0:
                arcpy.AddMessage("Beginning Linear Tools: ")
                QAQCcodeReset_line.QAQCcodeReset_tool(input_gdb,work_area)
                wetlandTypeCalc_line.wetlandTypeCalc(input_gdb,work_area)
                create_NWIID.create_NWIID_tool(input_gdb,work_area,"line")
                
                incorrectWetlandsCodes_line.incorrectCodes(input_gdb,work_area)
                linearPolyOverlaps_line.linearPolyOverlaps_tool(input_gdb,work_area)
                sliverWetlands_line.sliverWetlands(input_gdb,work_area)
                linearOverlaps.overlappingWetlands(input_gdb,work_area)
                linearGaps_line.linear_gaps_tool(input_gdb,work_area)
        
        #Riparian tools
        if arcpy.Exists(rip_poly):

            riparian_count = arcpy.management.GetCount(rip_poly).getOutput(0)
            if int(riparian_count) > 0:
            
                arcpy.AddMessage("Beginning Riparian Tools: ")

                QAQCcodeReset_Riparian.QAQCcodeReset_tool(input_gdb,work_area)
                riparianTypeCalc.riparianTypeCalc(input_gdb,work_area)
                create_NWIID.create_NWIID_tool(input_gdb,work_area,"rip")

                sliverRiparian.sliverRiparian_tool(input_gdb,work_area)
                adjacentRiparian.adjacentRiparian(input_gdb,work_area)
                overlappingRiparian.overlappingWetlands(input_gdb,work_area)
                sliverUplands_Riparian.sliverUplands(input_gdb,work_area)
                incorrectRiparianCodes.incorrectCodes(input_gdb,work_area)

        #Summary Tools
        arcpy.AddMessage("Beginning Summary Tools: ")
        QAQCSummary_poly.QAQCSummary_tool(input_gdb,work_area,verifier_name,save_summary_table)
        if arcpy.Exists(rip_poly):
            QAQCSummary_Riparian.QAQCSummary_tool(input_gdb,work_area,verifier_name,save_summary_table)

        return

class create_NWIID(object):
    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "Create NWI_ID"
        self.description = "This tool calculates the unique NWI_ID code. \n This tool needs to be run before data is submitted to NWI for QA."
        self.canRunInBackground = False
        self.category = "Individual Polygonal Tools"

    
    def getParameterInfo(self):
        """Define parameter definitions"""
        # params [0]
        input_gdb = arcpy.Parameter(name = "input_gdb",
                                   displayName = "Input Geodatabase",
                                   datatype = "DEWorkspace",
                                   parameterType = "Required", #Required|Optional|Derived
                                   direction = "Input", #Input|Output
                                   )

        work_area = arcpy.Parameter(name = "work_area",
                                     displayName = "Work Area",
                                     datatype = "String",
                                     parameterType = "Required", #Required|Optional|Derived
                                     direction = "Input", #Input|Output
                                     )
        work_area.value = "CONUS"
        work_area.filter.type = "ValueList"
        work_area.filter.list = ["CONUS","AK","HI","PRVI","PacTrust"]

        #params = None
        return [input_gdb,work_area]

    def isLicensed(self):
        """Set whether tool is licensed to execute."""
        return True

    def updateParameters(self, parameters):
        """Modify the values and properties of parameters before internal
        validation is performed.  This method is called whenever a parameter
        has been changed."""

        return

    def updateMessages(self, parameters):
        """Modify the messages created by internal validation for each tool
        parameter.  This method is called after internal validation."""

        return

    def execute(self, parameters, messages):
        """The source code of the tool."""

        arcpy.AddMessage("Create NWI_ID Tool Version 1.4.3")
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        
        arcpy.management.Compact(input_gdb)

        '''
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        qaqc_code = "'NNNNNN'"
        arcpy.management.CalculateField(wet_poly,"QAQC_CODE",qaqc_code,"PYTHON_9.3")
        '''
        try:
            create_NWIID.create_NWIID_tool(input_gdb,work_area,"wet")
        except arcpy.ExecuteError:
            arcpy.AddMessage(arcpy.GetMessages())
        return

    @staticmethod
    def create_NWIID_tool(arg1,arg2,arg3):
        input_gdb = str(arg1)
        work_area = str(arg2)
        wet_rip = str(arg3)
        if wet_rip == "line":
            wet_rip2 = "wet"
        else:
            wet_rip2 = wet_rip
            
        dataset_name = {"wet":"wetlands",
                        "rip":"riparian",
                        "line":"wetlands"}
        type_name_dict = {"wet":"poly",
                     "rip":"poly",
                     "line":"line"}

        wet_poly = os.path.join(input_gdb,"{work_area}_{feature_dataset}".format(work_area = work_area,feature_dataset=dataset_name[wet_rip]),"{work_area}_{wet_rip2}_{type_name}".format(work_area=work_area,wet_rip2=wet_rip2,type_name=type_name_dict[wet_rip]))
        #rip_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_rip_poly".format(work_area=work_area))

        #arcpy.management.CalculateField(wet_poly,"QAQC_CODE",qaqc_code,"PYTHON_9.3")
        listFields = []
        fields = arcpy.ListFields(wet_poly)
        for field in fields:
            listFields.append(field.name)
            
        #This section checks for existence of NWI_ID and GLOBALID in the schema.
        #If they do not exist, then it adds them. 
        if "NWI_ID" not in listFields:
            arcpy.management.AddField(wet_poly,"NWI_ID","TEXT","","",255)
            #arcpy.AddMessage("NWI_ID created")
            
        if "GLOBALID" not in listFields:
            arcpy.management.AddGlobalIDs(wet_poly)

        #This cursor checks for the existence of duplicate NWI_ID values. 
        fields = ['NWI_ID','GLOBALID']
        #cursor_start_time = time.time()
        nwi_id_list = []
        begin_dup_nwi_id_list = []
        with arcpy.da.SearchCursor(wet_poly,"NWI_ID") as cursor:
            for row in cursor:
                if row[0] in nwi_id_list:
                    if row[0] not in begin_dup_nwi_id_list:
                        begin_dup_nwi_id_list.append(row[0])
                nwi_id_list.append(row[0])
                
        #dup_nwi_id_list = {x for x in nwi_id_list if nwi_id_list.count(x) > 1}
        arcpy.AddMessage(f"The length of the duplicate NWI_ID list is {len(begin_dup_nwi_id_list)}.")
        #cursor_run_time = (time.time() - cursor_start_time)
        #arcpy.AddMessage(f"The cursor approach took {datetime.timedelta(seconds=cursor_run_time)}.")

        #This cursor calculates NWI_ID values for every feature that either
        #does not have a value, or is a duplicate. 
        edit = arcpy.da.Editor(input_gdb)
        edit.startEditing()
        edit.startOperation()
        with arcpy.da.UpdateCursor(wet_poly,fields) as cursor:
            #arcpy.AddMessage("creating NWI_ID...")
            current_time=datetime.datetime.now()
            YYYYMM = str(current_time.year)+'{:02d}'.format(current_time.month)
            for row in cursor:

                work_area_abbv_dict = {"CONUS":"CS",
                                       "AK":"AK",
                                       "HI":"HI",
                                       "PRVI":"PR",
                                       "PacTrust":"PT"
                                       }
                data_type_dict = {"wet":"w",
                                  "rip":"r",
                                  "line":"l"}
                
                NWI_ID = "{a}{b}{c}{d}_01".format(a=YYYYMM,b=work_area_abbv_dict[work_area],c=data_type_dict[wet_rip],d=str(row[1]))
                #arcpy.AddMessage(f"current NWI_ID[:10] is {row[0][:10]}")
                if row[0] is None:
                    row[0] = NWI_ID
                    cursor.updateRow(row)
                if len(row[0]) < 10:
                    row[0] = NWI_ID
                    cursor.updateRow(row)
                if row[0] in begin_dup_nwi_id_list:
                    #arcpy.AddMessage("This NWI_ID is a duplicate and needs to be overwritten.")
                    row[0] = NWI_ID
                    cursor.updateRow(row)
                if row[0][8] != 'w':
                    newID = row[0][:8]+'w'+row[0][9:]
                    row[0] = row[0][:8]+'w'+row[0][9:]
                    #arcpy.AddMessage(f"New ID: {newID}")
                    cursor.updateRow(row)
                                             
                    
                
        edit.stopOperation()
        edit.stopEditing(True)

        '''
        freq_start_time = time.time()
        #Frequency of NWI_ID?
        unique_IDs = os.path.join(input_gdb,"UniqueNWIIDs")
        duplicate_IDs = os.path.join(input_gdb,"DuplicateUniqueNWIIDs")
        arcpy.analysis.Frequency(wet_poly, unique_IDs, "NWI_ID", None)
        arcpy.conversion.ExportTable(unique_IDs,duplicate_IDs,"FREQUENCY > 1")
        freq_run_time = (time.time() - freq_start_time)
        arcpy.AddMessage(datetime.timedelta(seconds=freq_run_time))
        '''
        
        #Cursor Approach?
        #cursor_start_time = time.time()
        nwi_id_list = []
        dup_nwi_id_list = []
        with arcpy.da.SearchCursor(wet_poly,"NWI_ID") as cursor:
            for row in cursor:
                if row[0] in nwi_id_list:
                    if row[0] not in dup_nwi_id_list:
                        dup_nwi_id_list.append(row[0])
                nwi_id_list.append(row[0])
                
        #dup_nwi_id_list = {x for x in nwi_id_list if nwi_id_list.count(x) > 1}
        arcpy.AddMessage(f"After running Create_NWI_ID, the length of the duplicate NWI_ID list is {len(dup_nwi_id_list)}.")
        #cursor_run_time = (time.time() - cursor_start_time)
        #arcpy.AddMessage(f"The cursor approach took {datetime.timedelta(seconds=cursor_run_time)}.")
        return

class create_NWIID_rip(object):
    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "Create NWI_ID"
        self.description = "This tool calculates the unique NWI_ID code. \n This tool needs to be run before data is submitted to NWI for QA."
        self.canRunInBackground = False
        self.category = "Individual Riparian Tools"

    
    def getParameterInfo(self):
        """Define parameter definitions"""
        # params [0]
        input_gdb = arcpy.Parameter(name = "input_gdb",
                                   displayName = "Input Geodatabase",
                                   datatype = "DEWorkspace",
                                   parameterType = "Required", #Required|Optional|Derived
                                   direction = "Input", #Input|Output
                                   )

        work_area = arcpy.Parameter(name = "work_area",
                                     displayName = "Work Area",
                                     datatype = "String",
                                     parameterType = "Required", #Required|Optional|Derived
                                     direction = "Input", #Input|Output
                                     )
        work_area.value = "CONUS"
        work_area.filter.type = "ValueList"
        work_area.filter.list = ["CONUS","AK","HI","PRVI","PacTrust"]

        #params = None
        return [input_gdb,work_area]

    def isLicensed(self):
        """Set whether tool is licensed to execute."""
        return True

    def updateParameters(self, parameters):
        """Modify the values and properties of parameters before internal
        validation is performed.  This method is called whenever a parameter
        has been changed."""

        return

    def updateMessages(self, parameters):
        """Modify the messages created by internal validation for each tool
        parameter.  This method is called after internal validation."""

        return

    def execute(self, parameters, messages):
        """The source code of the tool."""

        arcpy.AddMessage("Create NWI_ID Riparian Version 1.4.3")
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        
        arcpy.management.Compact(input_gdb)

        '''
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        qaqc_code = "'NNNNNN'"
        arcpy.management.CalculateField(wet_poly,"QAQC_CODE",qaqc_code,"PYTHON_9.3")
        '''
        try:
            create_NWIID_rip.create_NWIID_tool_rip(input_gdb,work_area,"rip")
        except arcpy.ExecuteError:
            arcpy.AddMessage(arcpy.GetMessages())
            
        return

    @staticmethod
    def create_NWIID_tool_rip(arg1,arg2,arg3):
        input_gdb = str(arg1)
        work_area = str(arg2)
        wet_rip = str(arg3)
        dataset_name = {"wet":"wetlands",
                        "rip":"riparian"}
        type_name_dict = {"wet":"poly",
                     "rip":"poly",
                     "line":"line"}

        '''
        unique_IDs = os.path.join(input_gdb,"UniqueNWIIDs")
        duplicate_IDs = os.path.join(input_gdb,"DuplicateUniqueNWIIDs")

        arcpy.analysis.Frequency(wet_poly, unique_IDs, "NWI_ID", None)
        arcpy.conversion.ExportTable(unique_IDs,duplicate_IDs,"FREQUENCY > 1")
        '''
        
        wet_poly = os.path.join(input_gdb,"{work_area}_{feature_dataset}".format(work_area = work_area,feature_dataset=dataset_name[wet_rip]),"{work_area}_{wet_rip}_{type_name}".format(work_area=work_area,wet_rip=wet_rip,type_name=type_name_dict[wet_rip]))
        #rip_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_rip_poly".format(work_area=work_area))

        #arcpy.management.CalculateField(wet_poly,"QAQC_CODE",qaqc_code,"PYTHON_9.3")
        listFields = []
        fields = arcpy.ListFields(wet_poly)
        for field in fields:
            listFields.append(field.name)
            
        #arcpy.AddMessage(listFields)

        if "NWI_ID" not in listFields:
            arcpy.management.AddField(wet_poly,"NWI_ID","TEXT","","",255)
            #arcpy.AddMessage("NWI_ID created")

        if "GLOBALID" not in listFields:
            arcpy.management.AddGlobalIDs(wet_poly)

        #This cursor checks for the existence of duplicate NWI_ID values. 
        fields = ['NWI_ID','GLOBALID']
        #cursor_start_time = time.time()
        nwi_id_list = []
        begin_dup_nwi_id_list = []
        with arcpy.da.SearchCursor(wet_poly,"NWI_ID") as cursor:
            for row in cursor:
                if row[0] in nwi_id_list:
                    if row[0] not in begin_dup_nwi_id_list:
                        begin_dup_nwi_id_list.append(row[0])
                nwi_id_list.append(row[0])
                
        #dup_nwi_id_list = {x for x in nwi_id_list if nwi_id_list.count(x) > 1}
        arcpy.AddMessage(f"The length of the duplicate NWI_ID list is {len(begin_dup_nwi_id_list)}.")
        #cursor_run_time = (time.time() - cursor_start_time)
        #arcpy.AddMessage(f"The cursor approach took {datetime.timedelta(seconds=cursor_run_time)}.")

        fields = ['NWI_ID','GLOBALID']

        edit = arcpy.da.Editor(input_gdb)
        edit.startEditing()
        edit.startOperation()
        with arcpy.da.UpdateCursor(wet_poly,fields) as cursor:
            #arcpy.AddMessage("creating NWI_ID...")
            current_time=datetime.datetime.now()
            YYYYMM = str(current_time.year)+'{:02d}'.format(current_time.month)
            for row in cursor:

                work_area_abbv_dict = {"CONUS":"CS",
                                       "AK":"AK",
                                       "HI":"HI",
                                       "PRVI":"PR",
                                       "PacTrust":"PT"
                                       }
                data_type_dict = {"wet":"w",
                                  "rip":"r",
                                  "line":"l"}
                
                NWI_ID = "{a}{b}{c}{d}_01".format(a=YYYYMM,b=work_area_abbv_dict[work_area],c=data_type_dict[wet_rip],d=str(row[1]))
                if row[0] is None:
                    row[0] = NWI_ID
                    cursor.updateRow(row)
                if len(row[0]) < 10:
                    row[0] = NWI_ID
                    cursor.updateRow(row)
                if row[0] in begin_dup_nwi_id_list:
                    row[0] = NWI_ID
                    cursor.updateRow(row)
                if row[0][8] != 'r':
                    row[0] = row[0][:8]+'r'+row[0][9:]
                    cursor.updateRow(row)
                
        edit.stopOperation()
        edit.stopEditing(True)

        #cursor_start_time = time.time()
        nwi_id_list = []
        dup_nwi_id_list = []
        with arcpy.da.SearchCursor(wet_poly,"NWI_ID") as cursor:
            for row in cursor:
                if row[0] in nwi_id_list:
                    if row[0] not in dup_nwi_id_list:
                        dup_nwi_id_list.append(row[0])
                nwi_id_list.append(row[0])
                
        #dup_nwi_id_list = {x for x in nwi_id_list if nwi_id_list.count(x) > 1}
        arcpy.AddMessage(f"After running Create_NWI_ID, the length of the duplicate NWI_ID list is {len(dup_nwi_id_list)}.")
        #cursor_run_time = (time.time() - cursor_start_time)
        #arcpy.AddMessage(f"The cursor approach took {datetime.timedelta(seconds=cursor_run_time)}.")

        return

class create_NWIID_line(object):
    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "Create NWI_ID"
        self.description = "This tool calculates the unique NWI_ID code. \n This tool needs to be run before data is submitted to NWI for QA."
        self.canRunInBackground = False
        self.category = "Individual Linear Tools"

    
    def getParameterInfo(self):
        """Define parameter definitions"""
        # params [0]
        input_gdb = arcpy.Parameter(name = "input_gdb",
                                   displayName = "Input Geodatabase",
                                   datatype = "DEWorkspace",
                                   parameterType = "Required", #Required|Optional|Derived
                                   direction = "Input", #Input|Output
                                   )

        work_area = arcpy.Parameter(name = "work_area",
                                     displayName = "Work Area",
                                     datatype = "String",
                                     parameterType = "Required", #Required|Optional|Derived
                                     direction = "Input", #Input|Output
                                     )
        work_area.value = "CONUS"
        work_area.filter.type = "ValueList"
        work_area.filter.list = ["CONUS","AK","HI","PRVI","PacTrust"]

        #params = None
        return [input_gdb,work_area]

    def isLicensed(self):
        """Set whether tool is licensed to execute."""
        return True

    def updateParameters(self, parameters):
        """Modify the values and properties of parameters before internal
        validation is performed.  This method is called whenever a parameter
        has been changed."""

        return

    def updateMessages(self, parameters):
        """Modify the messages created by internal validation for each tool
        parameter.  This method is called after internal validation."""

        return

    def execute(self, parameters, messages):
        """The source code of the tool."""

        arcpy.AddMessage("Create NWI_ID Linear Version 1.4.3")
        input_gdb = str(parameters[0].value)
        work_area = str(parameters[1].value)
        
        arcpy.management.Compact(input_gdb)

        '''
        wet_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_wet_poly".format(work_area=work_area))
        qaqc_code = "'NNNNNN'"
        arcpy.management.CalculateField(wet_poly,"QAQC_CODE",qaqc_code,"PYTHON_9.3")
        '''
        try:
            create_NWIID_line.create_NWIID_tool_line(input_gdb,work_area,"line")
        except arcpy.ExecuteError:
            arcpy.AddMessage(arcpy.GetMessages())
            
        return

    @staticmethod
    def create_NWIID_tool_line(arg1,arg2,arg3):
        input_gdb = str(arg1)
        work_area = str(arg2)
        wet_rip = str(arg3)
        if wet_rip == "line":
            wet_rip2 = "wet"
        else:
            wet_rip2 = wet_rip
            
        dataset_name = {"wet":"wetlands",
                        "rip":"riparian",
                        "line": "wetlands"}
        type_name_dict = {"wet":"poly",
                     "rip":"poly",
                     "line":"line"}

        '''
        unique_IDs = os.path.join(input_gdb,"UniqueNWIIDs")
        duplicate_IDs = os.path.join(input_gdb,"DuplicateUniqueNWIIDs")

        arcpy.analysis.Frequency(wet_poly, unique_IDs, "NWI_ID", None)
        arcpy.conversion.ExportTable(unique_IDs,duplicate_IDs,"FREQUENCY > 1")
        '''
        
        wet_poly = os.path.join(input_gdb,"{work_area}_{feature_dataset}".format(work_area = work_area,feature_dataset=dataset_name[wet_rip]),"{work_area}_{wet_rip2}_{type_name}".format(work_area=work_area,wet_rip2=wet_rip2,type_name=type_name_dict[wet_rip]))
        #rip_poly = os.path.join(input_gdb,"{work_area}_wetlands".format(work_area = work_area),"{work_area}_rip_poly".format(work_area=work_area))

        #arcpy.management.CalculateField(wet_poly,"QAQC_CODE",qaqc_code,"PYTHON_9.3")
        listFields = []
        fields = arcpy.ListFields(wet_poly)
        for field in fields:
            listFields.append(field.name)
            
        #arcpy.AddMessage(listFields)

        if "NWI_ID" not in listFields:
            arcpy.management.AddField(wet_poly,"NWI_ID","TEXT","","",255)
            #arcpy.AddMessage("NWI_ID created")
        
        if "GLOBALID" not in listFields:
            arcpy.management.AddGlobalIDs(wet_poly)

        #This cursor checks for the existence of duplicate NWI_ID values. 
        fields = ['NWI_ID','GLOBALID']
        #cursor_start_time = time.time()
        nwi_id_list = []
        begin_dup_nwi_id_list = []
        with arcpy.da.SearchCursor(wet_poly,"NWI_ID") as cursor:
            for row in cursor:
                if row[0] in nwi_id_list:
                    if row[0] not in begin_dup_nwi_id_list:
                        begin_dup_nwi_id_list.append(row[0])
                nwi_id_list.append(row[0])
                
        #dup_nwi_id_list = {x for x in nwi_id_list if nwi_id_list.count(x) > 1}
        arcpy.AddMessage(f"The length of the duplicate NWI_ID list is {len(begin_dup_nwi_id_list)}.")
        #cursor_run_time = (time.time() - cursor_start_time)
        #arcpy.AddMessage(f"The cursor approach took {datetime.timedelta(seconds=cursor_run_time)}.")
        fields = ['NWI_ID','GLOBALID']

        edit = arcpy.da.Editor(input_gdb)
        edit.startEditing()
        edit.startOperation()
        with arcpy.da.UpdateCursor(wet_poly,fields) as cursor:
            #arcpy.AddMessage("creating NWI_ID...")
            current_time=datetime.datetime.now()
            YYYYMM = str(current_time.year)+'{:02d}'.format(current_time.month)
            for row in cursor:

                work_area_abbv_dict = {"CONUS":"CS",
                                       "AK":"AK",
                                       "HI":"HI",
                                       "PRVI":"PR",
                                       "PacTrust":"PT"
                                       }
                data_type_dict = {"wet":"w",
                                  "rip":"r",
                                  "line":"l"}
                
                NWI_ID = "{a}{b}{c}{d}_01".format(a=YYYYMM,b=work_area_abbv_dict[work_area],c=data_type_dict[wet_rip],d=str(row[1]))
                if row[0] is None:
                    row[0] = NWI_ID
                    cursor.updateRow(row)
                if len(row[0]) < 10:
                    row[0] = NWI_ID
                    cursor.updateRow(row)
                if row[0] in begin_dup_nwi_id_list:
                    row[0] = NWI_ID
                    cursor.updateRow(row)
                if row[0][8] != 'l':
                    row[0] = row[0][:8]+'l'+row[0][9:]
                    cursor.updateRow(row)
                
        edit.stopOperation()
        edit.stopEditing(True)

        #cursor_start_time = time.time()
        nwi_id_list = []
        dup_nwi_id_list = []
        with arcpy.da.SearchCursor(wet_poly,"NWI_ID") as cursor:
            for row in cursor:
                if row[0] in nwi_id_list:
                    if row[0] not in dup_nwi_id_list:
                        dup_nwi_id_list.append(row[0])
                nwi_id_list.append(row[0])
                
        #dup_nwi_id_list = {x for x in nwi_id_list if nwi_id_list.count(x) > 1}
        arcpy.AddMessage(f"After running Create_NWI_ID, the length of the duplicate NWI_ID list is {len(dup_nwi_id_list)}.")
        #cursor_run_time = (time.time() - cursor_start_time)
        #arcpy.AddMessage(f"The cursor approach took {datetime.timedelta(seconds=cursor_run_time)}.")

        return

